% quick didactic example of how to implement a basic sign-flip permutation test
% to assess the null hypothesis that the Z statistic in the sample is drawn from
% a distribution centered on 0.
%
% J Carlin, MRC CBU, 20 February 2017

% make up some data
nsub = 20;
nvox = 100;
noise = randn(nsub,nvox);
effectsize = .5;
signal = rand(nsub,nvox) * effectsize;
data = signal + noise;

nperm = 1000;

% true stat - Z
truestat = mean(data) ./ std(data);

nulldist = NaN([nperm,nvox]);
wasgt = NaN([nperm,nvox]);

% we insert the true permutation. In general this should always be done
% (otherwise, p=0 is possible), because the true stat is part of the
% distribution we wish to estimate (it happened, so it's actually the best idea
% we have of what's really going on)
nulldist(1,:) = truestat;
wasgt(1,:) = 1;

for p = 2:nperm
    doflip = rand(nsub,1) > .5;
    % flip some test stats
    permdata = data;
    permdata(doflip,:) = permdata(doflip,:) .* -1;
    % estimate a null z stat
    nulldist(p,:) = mean(permdata) ./ std(permdata);
    wasgt(p,:) = nulldist(p,:) >= truestat;
end

maxdist = max(nulldist,[],2);

% vox p - what proportion of null stats exceeded the true stat (because the true
% stat is in there, this can't be better than 1/nperm)
voxp = sum(wasgt) / nperm;
% a less awkward way to construct the same thing would have been this, if you're
% comfortable with bsxfun
% voxp2 = sum(bsxfun(@ge,nulldist,truestat)) / nperm;

% maximal stat p - what proportion of the MAX stats exceed the true stat (so
% unlike the above, one distribution for all voxels)
fwep = (sum(repmat(maxdist,[1 nvox]) >= repmat(truestat,[nperm,1]))) / nperm;
% a less awkward solution would have been
% fwep2 = sum(bsxfun(@ge,maxdist,truestat)) / nperm;

[bestp,ind] = min(voxp);
figure(10);
clf(10);
subplot(2,1,1);
hist(nulldist(:,ind),round(nperm/20));
line(truestat(:,[ind ind])',ylim,'color','r')
th = text(truestat(:,ind),max(ylim),sprintf('true stat (p=%.3f)',bestp),...
    'horizontalalignment','center','verticalalignment','bottom');
ylabel('frequency');
xlabel('Z');
box off
set(gca,'tickdir','out','ticklength',get(gca,'ticklength')*2);
subplot(2,1,2);
hist(maxdist,round(nperm/20));
line(truestat(:,[ind ind])',ylim,'color','r')
th = text(truestat(:,ind),max(ylim),sprintf('true stat (p(FWE)=%.3f)',...
    fwep(ind)),'horizontalalignment','center','verticalalignment','bottom');
ylabel('frequency');
xlabel('max(Z)');
box off
set(gca,'tickdir','out','ticklength',get(gca,'ticklength')*2);

% print
printsize = [8,12];
set(gcf,'units','centimeters','paperunits','centimeters',...
    'papersize',printsize,'paperposition',[0 0 printsize]);
print(fullfile(fileparts(mfilename('fullpath')),'permutationdemo.pdf'),'-dpdf');
