% interactive demonstration of how different experimental designs influence
% design efficiency and variance inflation factor.
%
% If the input onsets is undefined the user can specify onsets graphically with
% mouse clicks.
%
% CAVEATS: this is a linear-assumptions solution, so we can't capture some
% effects that are likely to limit the efficiency of particular designs. For
% instance, in the real brain, there is almost certainly saturation of the HRF
% as you put experimental events too closely together.
%
% INPUTS:
% onsets - see convolveonsets
% conind - see convolveonsets (nb, only 2 regressors allowed)
% convec ([1 -1]) - contrast vector
% nvol (90) - number of volumes
%
% OUTPUTS:
% eff - efficiency (scalar)
% v - variance inflation factor (one per regressor)
%
% 20180207 J Carlin, MRC CBU
%
% [eff,v] = practical_efficiency(onsets,conind,convec,nvol)
function [eff,v] = practical_efficiency(onsets,conind,convec,nvol)

% ensure code is on path
if isempty(which('convolveonsets')) || isempty(which('spm_spm'))
    error('code not on path - add spm12, pilab, johansmatlab');
end

% input parse
if ~exist('nvol','var') || isempty(nvol)
    nvol = 90;
end

screendims = get(0,'screensize');
screendims = [screendims(3), screendims(4)*.9];
tr = 2;
dur = .5;

if ~exist('onsets','var') || isempty(onsets)
    assert(~exist('conind','var') || isempty(conind),...
        'conind must be undefined if onsets is undefined');
    F2 = figure(200);
    set(F2,'position',[1,1,screendims(1)/2 screendims(2)]);
    clf(F2);
    imagesc(zeros(nvol,2));
    ylabel(sprintf('volumes (tr=%.2fs)', tr));
    xlabel('regressors');
    set(gca,'xtick',[.5,1.5,2.5]);
    grid on;
    fprintf('click with the mouse to generate events in the design matrix.\n');
    fprintf('press return when you are finished.\n');
    [conind,onsets] = getptsNoDoubleClick;
    conind = round(conind);
    conind(conind<1) = 1;
    conind(conind>2) = 2;
    onsets(onsets<0) = 0;
    onsets(onsets>nvol) = nvol;
end
if ~exist('convec','var') || isempty(convec)
    convec = [1 -1];
end
assert(isequal(unique(conind(:)),[1,2]'),'must specify 2 conditions');
assert(size(convec,2)==2,'contrast vector must have 2 columns');

% generate convolved design matrix - nb, scaling the design matrix throws the
% efficiency calculation, so must disable this
X = convolveonsets(onsets,conind,tr,nvol,'dur',dur,'scalepeak',0);

% high-pass filter with 128s period
Xf = spm_filter(struct('RT',tr,'row',1:nvol,'HParam',128),X);

eff = designefficiency(Xf,convec);
v = vif(Xf);

% make figure
F = figure(100);
set(F,'position',[screendims(1)/2,1,screendims(1)/2 screendims(2)]);
clf(F);
ax = subplot(1,5,1:2);
imagesc(X);
ylabel(sprintf('volumes (tr=%.2fs)', tr));
xlabel('regressors');
title({'raw design matrix',''});
box off;

subplot(1,5,3:4);
imagesc(Xf);
xlabel('regressors');
% with 2 conditions, vif(1) == vif(2)
title({'filtered design matrix',...
    sprintf('eff=%.0f, vif=%.2f',eff,v(1))});
box off;
colormap(gray(256));

subplot(1,5,5);
contime = Xf * convec';
plot(contime,1:nvol);
set(gca,'ydir','reverse');
% place grids at this spacing
gspace = 30;
m = ceil(max(abs(contime)) / gspace) * gspace;

set(gca,'xtick',[-m:gspace:m]);
set(gca,'xlim',[-m,m]);
grid on
title({'contrast time course',''});
