% example answer for first level GLM practical. We load up some real fMRI data,
% and fit it using increasingly sophisticated modeling approaches.
%
% 2017 J Carlin, MRC CBU, Cambridge, UK
% johan.carlin@mrc-cbu.cam.ac.uk

clear all
close all

% presenter settings for visibility - you probably don't need this
% set(0,'defaultaxesfontsize',16,'defaulttextfontsize',16,'defaultlinelinewidth',2)

fprintf('\nINSPECT DATA\n')
% we assume you are running from a directory where these files are available
rundata = load('rundata');
% rundata is a struct array with one entry per run, and fields for data,
% designmatrix, and trends (the scanner drift model)
rundata = rundata.rundata

parameters = load('parameters');
% parameters is a struct that keeps track of some shared settings
parameters = parameters.parameters

% first, let's plot the data from one run
figure(100);
clf(100);
subplot(1,3,1);
imagesc(rundata(1).data);
title('data');
xlabel('voxels');
ylabel('time (scans)')
subplot(1,3,2);
imagesc(rundata(1).designmatrix);
set(gca,'xtick',1:numel(parameters.conditions),...
    'xticklabel',parameters.conditions);
title('design');
xlabel('regressors');
subplot(1,3,3);
imagesc(rundata(1).trends);
xlabel('covariates');
title('trend model');

fprintf('\nNAIVE MODEL FIT\n')
% so let's start out with a really naive approach - let's just fit the task model
% directly to the data
% key function to understand: vertcat takes a bunch of data (here runs) and
% concatenates into a single long matrix
alldata = vertcat(rundata.data);
alldesign = vertcat(rundata.designmatrix);
% we will add a constant because otherwise this gets really contrived
alldesign(:,end+1) = 1;
% OLS fit
b = alldesign \ alldata;
% predicted time courses and residual errors
fitted = alldesign * b;
res = alldata - fitted;
% explained variance (squared r) for the model fit
r2 = corrpairs(alldata,fitted).^2;
% quartiles and median explained variance
quartiles_raw = prctile(r2,[25 50 75])
% find the best performing voxel
[maxr2_raw,ind] = max(r2)
% and let's visualise it
figure(200);
clf(200);
ph(1) = plot(alldata(:,ind),'.-');
hold all
ph(2) = plot(fitted(:,ind),'-','linewidth',2);
l = legend(ph,{'data','fit'});
title('best fitting voxel (raw fit)')
xlabel('time (tr)')
ylabel('image intensity')
% bit hard to see what's going on unless we restrict the axis
xlim([0 200]);
% so that works surprisingly well, but it's pretty clear that our model needs to
% be improved
%
% CLASS: What can we do to improve things?

% let's convolve the design matrix with the hrf, which looks like this
fprintf('\nCONVOLVE THE DESIGN MATRIX\n')
figure(300);
clf(300);
plot(parameters.hrf);
xlabel('time (scan)');
ylabel('intensity');
title('hemodynamic response function');

for sess = 1:numel(rundata)
    % CLASS: Why do we convolve separately for each run? Because we wouldn't
    % want the predicted HRF to spill over into the next run.
    rundata(sess).designconv = conv2(rundata(sess).designmatrix,parameters.hrf);
    % this is a bit silly but we need to trim off the excess
    nsamples = size(rundata(sess).designmatrix,1);
    rundata(sess).designconv = rundata(sess).designconv(1:nsamples,:);
end
% so now our design matrix looks like this
figure(400);
clf(400);
subplot(2,1,1);
plot(rundata(1).designmatrix);
title('original')
subplot(2,1,2);
clear ph
ph = plot(rundata(1).designconv);
lh = legend(ph,parameters.conditions);
xlabel('time (scan)');
title('convolved')

fprintf('\nCONVOLVED MODEL FIT\n')
% CLASS: do you think our fit is better now? Let's see if you can perform the
% fit just as above by re-using the code
% use the convolved design matrix
alldesign = vertcat(rundata.designconv);
% we will add a constant because otherwise this gets really contrived
alldesign(:,end+1) = 1;
% OLS fit
b = alldesign \ alldata;
% predicted time courses and residual errors
fitted = alldesign * b;
res = alldata - fitted;
% explained variance (squared r) for the model fit
r2 = corrpairs(alldata,fitted).^2;
% quartiles and median explained variance
quartiles_conv = prctile(r2,[25 50 75])
% find the best performing voxel
[maxr2_conv,indconv] = max(r2)
% and let's visualise it
figure(500);
clf(500);
clear ph
ph(1) = plot(alldata(:,indconv),'.-');
hold all
ph(2) = plot(fitted(:,indconv),'-','linewidth',2);
l = legend(ph,{'data','fit'});
title('best fitting voxel (convolved fit)')
xlabel('time (tr)')
ylabel('image intensity')
% bit hard to see what's going on unless we restrict the axis
xlim([0 200]);

% So that's better, but something is still a little off... Perhaps if we plot
% the residual time course?
figure(600);
clf(600);
plot(res(:,indconv));
title('residuals (convolved)');
% CLASS: what might be wrong here? Well, it looks like there are slow trends in
% the data, so having a single constant isn't quite doing the job. We need a
% more sophisticated trend model.

fprintf('\nDETREND DATA AND MODEL\n')
% We need to detrend! Back to the drawing board.
for sess = 1:numel(rundata)
    % CLASS: why do we do this separately for each run? Because the drift might
    % go in different directions on different runs.
    rundata(sess).datadet = projectout(rundata(sess).data,rundata(sess).trends);
    rundata(sess).designdet = projectout(rundata(sess).designconv,rundata(sess).trends);
end

% what did that do anyway? Well, let's look at the data
figure(700);
clf(700);
subplot(1,2,1);
imagesc(rundata(1).data);
title('raw');
ylabel('time (tr)');
xlabel('voxels');
subplot(1,2,2);
imagesc(rundata(1).datadet);
xlabel('voxels');
title('detrended');
% almost as good as physiology! (which is also typically heavily high-pass filtered btw)

% CLASS: let's repeat the linear fit a third time and see what we get!
% use the detrended design matrix
fprintf('\nDETRENDED MODEL FIT\n')
alldesign = vertcat(rundata.designdet);
% and the detrended data - a common error is to detrend only one side of the
% equation...
alldata = vertcat(rundata.datadet);
% we no longer need a constant - the data and design are mean 0
% alldesign(:,end+1) = 1;
% OLS fit
b = alldesign \ alldata;
% predicted time courses and residual errors
fitted = alldesign * b;
res = alldata - fitted;
% explained variance (squared r) for the model fit
r2 = corrpairs(alldata,fitted).^2;
% quartiles and median explained variance
quartiles_detrend = prctile(r2,[25 50 75])
% find the best performing voxel
[maxr2_detrend,inddet] = max(r2)
% and let's visualise it
figure(800);
clf(800);
clear ph
ph(1) = plot(alldata(:,inddet),'.-');
hold all
ph(2) = plot(fitted(:,inddet),'-','linewidth',2);
l = legend(ph,{'data','fit'});
title('best fitting voxel (detrended fit)')
xlabel('time (tr)')
ylabel('image intensity')
% bit hard to see what's going on unless we restrict the axis
xlim([0 200]);

% Looks better, no?
% indeed, is we plot the residuals they now look nice and flat
figure(900);
clf(900);
plot(res(:,indconv))
title('residuals (detrended)')

fprintf('\nEQUIVALENCE OF DETRENDING AND COVARIATE GLM FIT\n')
% (by the way, what was all that projection about? It turns out, it's equivalent
% to just fitting with the trend model in the design matrix, like this
balt = [alldesign vertcat(rundata.trends)] \ vertcat(rundata.data);
% we now have lots more parameter estimates (for the trends), but the task betas
% are the same (down to rounding error)
b_detrend = b(1:4,1)
b_covariate = balt(1:4,1)
% so detrending is convenient and speeds up fits a bit (smaller matrices to
% invert)

fprintf('\nAUTOCORRELATION ANALYSIS\n')
% speaking of residuals. What was that thing we heard about autocorrelation of
% the residuals?
r1 = corrpairs(res(1:end-1,:),res(2:end,:));
medr1 = median(r1)
r2 = corrpairs(res(1:end-2,:),res(3:end,:));
medr2 = median(r2)
r3 = corrpairs(res(1:end-3,:),res(4:end,:));
medr3 = median(r3)
figure(1000);
clf(1000);
plot(res(1:end-1,inddet),res(2:end,inddet),'.');
lsline;
xlabel('time')
ylabel('time+1')
title(sprintf('residual autocorrelation, r=%.2f',r1(inddet)));

% so we need an autoregressive model to get valid p values and standard errors.
% The betas themselves are probably ok though! We can also see a hint here of
% why the AR(1) model (which only corrects first-order autocorrelation) might
% not be adequate (see Eklund et al., 2012, NeuroImage) - there's substantial
% autocorrelation 2 steps removed as well!
