%%
% Note: 
% Typing "help <matlab command>" shows details about Matlab commands and their usage
% Pressing "F1" open the Matlab Help window (also from main window: Help -> Matlab Help, ...
% ...which is searchable and contains more information, examples etc.
% Ask Google, or look at the Mathworks page (e.g. http://www.mathworks.com/help/techdoc/index.html)
% This script uses "Cell" mode (i.e. %% to mark different sections)
% You can mark sections in the script and execute them pressing "F9" (note some bits only work if did the bits before)
% The example data are in /imaging/olaf/MEG/MatlabBeginners
% Enjoy!
% OH, Jan 2011, Nov 2012

%% EMEG data

%% Read EMEG data
load EMEGdata.mat;  % data for this presentation are stored in a Matlab file; the data are in a structure "data" (see workspace)

%% Type the following expression and press enter:
data    % data structure
data.info   % information about data and acquisition parameters

data.info.ch_names  % names of EEG, MEG and other recording channels
data.info.chs(1).ch_name
data.info.chs(307).ch_name

%% Extract the data for analysis into a matrix "eegdata"
eegdata = data.evoked.epochs( 307:379, : ); % get EEG channels
eegdata(61:63,:) = [];  % delete non-EEG (EOG, ECG)

[nr_ch, nr_tpts] = size( eegdata );     % determine number of rows (channels) and columns (time points) of EEG data matrix

%% plot only EEG
plot(eegdata');

% add latencies to plot
lats = 1000*data.evoked.times;  % sample latencies from s to ms
plot(lats, eegdata');           % plot with latencies

% Try the Data Cursor and Zoom function in the Matlab figure


%% Find channel with maximum amplitude
[maxval, maxchan] = max( max( abs(eegdata') ) );    % maximum of absolute values in matrix
% plot channel with maximum amplitude
plot( lats, eegdata(maxchan,:) );

%% Find how many data points are above half of maximum
maxchandat = eegdata(maxchan,:);        % get time course of maximum channel
maxpts = find( maxchandat > maxval/2 ); % find all time points above half-maximum amplitude
length( maxpts )                        % count number of time points

%% baseline correction using FOR-loop
% Complicated (but possibly educating) version of baseline correction
% Two nested FOR-loops
for cc=1:nr_ch, % for all channels...
    basemean = mean( eegdata(cc,1:33) );         % mean baseline amplitude at channel cc
    for tt = 1:nr_tpts,                                 % for all time points...
        eegdata2(cc,tt) = eegdata(cc,tt) - basemean;    % subtract baseline value at each time point at each channel
    end;
end;
figure; % new figure
plot( lats, eegdata2' );

%% baseline correction using matrices; No FOR-loop
clear eegdata2; % clear previous result
MeanBaseline = mean( eegdata(:,1:33), 2 );     % mean baseline for all channels (along rows), the result is a column vector
RepMeanBaseline = repmat( MeanBaseline, 1, nr_tpts );     % the result is a matrix, same dimensions as data matrix
eegdata2 = eegdata - RepMeanBaseline;                    % subtract all baseline values from all corresponding channels and time points
figure;
plot( lats, eegdata2' );


%% average reference for EEG using for-loop
% see PPT for illustration

% Complicated (but possibly educating) version; two FOR-loops
clear eegdata2;
for tt = 1:nr_tpts,
    MeanOfColumn = mean( eegdata(:,tt) );                   % mean value across channels at timepoint tt
    for cc = 1:nr_ch,     
        eegdata2(cc,tt) = eegdata(cc,tt) - MeanOfColumn;    % subtract mean at all time points at all channels
    end;
end;
figure;
plot( lats, eegdata2' );

% avg reference using matrices
MatrixOperator = eye(nr_ch) - ones(nr_ch)/nr_ch;      % average reference "operator"; identity matrix minus ones-matrix divided by number of channels (see PPT)
eegdata2 = MatrixOperator * eegdata;                  % multiply data with operator


%% compute SNRs (stdev)
StdBaseline = std( eegdata2(:,1:33), 0, 2 );   % vector of standard deviations of baseline for each channel
RepStdBaseline = repmat( StdBaseline, 1, nr_tpts );      % the result is a matrix, same dimensions as data matrix
eegdatasnr = eegdata2 ./ RepStdBaseline;                 % divide each channel by corresponding standard deviation of baseline
plot( lats, eegdatasnr' );


%% compute variance, covariance, and correlation
covmat = cov( eegdata' );                           % covariance matrix over time
imagesc( covmat );                                  % visualise matrix
colorbar;                                           % include colour scale bar

% change to symmetrical colour scaling:
maxval = max( max( abs( covmat ) ) );               % maximum value in covariance matrix (for scaling)
figure;
imagesc( covmat, [-maxval maxval] );                % specify scaling interval
colorbar;

varvec = var(eegdata, 0, 2);    % vector with variances over time (equal to diag(covmat))

% Correlation
[r,p] = corrcoef( eegdata' );                       % get correlation coefficients between channels over time and their p-values
imagesc(r, [-1 1]);                                 % visualise correlation coefficients
title('Correlation Coefficients');
colorbar;
figure;
imagesc((1-p), [0 1]);                              % visualise p-values
title('P-Values');
colorbar;

%% PCA
[coeff, score, latent] = princomp( eegdata2 );       % compute PCA of data matrix 
plot( latent );                                     % plot factor scores
xlabel('Component');
ylabel('Eigenvalue');
% You see that factors scores fall off steeply, and are already close to zero around the 10th component


%% Stats

% simple: mean, std, median

% ttest against zero
[H,P,CI,STATS] = ttest( eegdata(1,:)', 0, 0.05, 'both' );  % test all time points against zero (doesn't make much sense; just for illustration)
% H only tells you whether null hypothesis rejected or not
P           % contains p-values
STATS       % contains t-values, degrees of freedom

% paired t-test
[H,P,CI,STATS] = ttest( eegdata(1,:)', eegdata(2,:)', 0.05, 'both' );  % paired t-test between channels 1 and 2 (doesn't make much sense; just for illustration)

% for unpaired t-tests, use ttest2
% see also: anova1, anova2, anovan (not repeated measures)


%% input/output
% there are convenient Matlab functions for reading/writing matrices and Matlab variables

% load/save (if everything in Matlab format)
load SPM.mat
% if you've changed something, save it again:
save SPM.mat SPM

% write matrices
subject_number = 1;                                             % assume you are processing the first of many subjects
filename = ['data_subject' num2str(subject_number) '.txt'];     % create the file name for this specific subject
dlmwrite(filename, eegdata, ' ');                               % write data matrix to text file (look at it in a text editor)

% read numbers
eegdata2 = dlmread(filename);                                   % read the matrix again

% Writing correlation matrix (from above) to Excel file (only works under Windows, not in Linux)
filename = ['data_subject' num2str(subject_number) '.xls'];
xlswrite( filename, r );
xlswrite( filename, p, 2 ); % worksheet 2

%% More flexibility with commands "fprintf", "fscanf", "textread"
% reading/writing text can be a bit tricky

TestData = rand(10,3);    % 10-by-3 matrix of random numbers

MyLabels = {'Prime', 'Target', 'Control'};
MyFile_id = fopen('TestFile.txt', 'w');    % create a file for writing

fprintf(MyFile_id, '%s %s %s\n', MyLabels{1}, MyLabels{2}, MyLabels{3});

for ii = 1:10,
    fprintf(MyFile_id, '%f %f %f\n', TestData(ii,:));
end;

fclose(MyFile_id);



%% Possible "Exercises"

% 1
% compute the correlation matrix for different columns of the fMRI design matrix
% Find the columns that show maximum correlation
% Visualise the matrix (incl. colorbar)
% Note that columns 9-14 and 23-28 correspond to movement parameters (and 29-30 to the session means)

% 2
% Compute RMS curves for EEG, Gradiometers and Magnetometers, and plot them into the same graph (add latencies, labels and legend)
% Because the three data types have different physical units (Volts, Tesla,Tesla/m), you should normalise them in some way (e.g. to maximum, or compute SNRs)

% 3
% When contrasting two conditions against each other in fMRI (e.g. words-control), the elements of the contrast vector should add up to zero (e.g. [1, -1])
% Check whether this is the case for the design matrix in above example
% (e.g. compute the sum for each contrast vector, and display the values together with their names)

% 4
% Compute separate PCAs for data from EEG, Gradiometers and Magnetometers
% Plot the 3 eigenvalue curves (as above) in one graph, indicating the data type in the legend
% (note: the eigenvalue vectors will be of different length; you will have to pad the shorter ones with zeros)

% 5
% Write the EEG, Gradiometer and MEG data into different sheets of the same Excel workbook
% Because Excel worksheets have limited width, you may want to write them with electrodes as columns and time points as rows (i.e. transposed)
% Because these data sets are quite large, you may want to output only every second time point
% You may want to add the latencies as the first column of each worksheet (i.e. include it as the first column of the output matrix)


%% Script for multiple data sets

% The data are in 3 Matlab files (subj1_emeg.mat etc.)
% each *.mat file contains a structure "data"
% The following script will compute RMS values for magnetometers for each subject, plot the curves into one plot, and output the result as a matrix
% (note: the data values are very small, because they are in the femto-Tesla range, expressed in units of Tesla)

path_data = '/imaging/olaf/MEG/MatlabBeginners/';   % directory where data are
subject_names = {'subj1', 'subj2', 'subj3'};        % names or your subjects
filename_out = '/YourDirectory/YourFilename.txt';   % Filename for output RMS data

nr_subs = length( subject_names );                  % number of subjects as specified above

colors = {'b', 'g', 'r', 'k', 'c', 'm'};            % list of possible colours for plotting lines (see "help plot")

figure; % create a new figure
magdata = [];   % Initialise the data matrix (and make sure it's empty to begin with)
for ss = 1:nr_subs, % for each subject...
    filename_in = fullfile(path_data, [subject_names{ss} '_emeg.mat']); % data file to read for this subject
    load( filename_in );
    
    magdata(:,:,ss) = data.evoked.epochs( 3:3:306, : ); % Magnetometer data, read into 3D matrix (channels, time points, subjects)
    
    magdatarms(:,ss) = sqrt( mean( magdata(:,:,ss).^2 ) );    % Keep RMS values for each subject        
    
    lats = 1000*data.evoked.times;  % sample latencies 
    
    ph = plot( lats, magdatarms(:,ss), colors{ss} );    % Plot RMS curve for this subject in individual colour; use plot handle to modify later
    hold on;    % keep plot while plotting the next line
    set(ph, 'LineWidth', 2);    % change line width in plot
end;
grid on;    % plot data with grid
legend( subject_names );    % so you know which line belongs to which subject
th = title('RMS Curves for Mags');   % Use figure handle to modify later
xh = xlabel('Latency (ms)');
yh = ylabel('RMS (fT)');
set(th, 'FontSize', 18, 'FontWeight', 'Bold');
set(xh, 'FontSize', 18);
set(yh, 'FontSize', 18);

% Not you are so proud of your figure that you want to save it for posterity...
% This bit will save it using the output file name, but adding the appropriate filename extensions
[filepath, filename, fileext] = fileparts(filename_out);    % get parts of output filename
figure_name_fig = fullfile( filepath, [filename '.fig']);
saveas(gcf, figure_name_fig, 'fig');    % save current figure in Matlab format (use it with "open")
figure_name_ai = fullfile( filepath, [filename '.ai']);
saveas(gcf, figure_name_ai, 'ai');    % save current figure in Illustrator format
figure_name_jpg = fullfile( filepath, [filename '.jpg']);
saveas(gcf, figure_name_jpg, 'jpg');    % save current figure in JPG format

meanmagdata = mean( magdata, 3 );   % compute average across subjects (3rd dimension)

output_mat = [lats', magdatarms];   % add latencies as first column
dlmwrite( filename_out, output_mat, ' ' );  % Write matrix with spaces as delimiter

%% Exercise
% Extend the script to process EEG and gradiometer data as well (you can put them into one structure, e.g.: data.eeg, data.mags, data.grads)
% for example, you can plot different figures for each sensor type (incl. 3 subjects)
% or plot different figures for each subject (incl. 3 channel types, but note: scaling)
% How strongly do RMS curves for different subjects and/or sensor types correlation with each other?
% How similar are covariance matrices for different subjects?
% Compare SNRs between sensor types



