% Vectorized examples

%% Get even numbers 2-20 
% myevennumbers=2*(1:10); % this is one option
myevennumbers=2:2:20; % this is even better

%% Generate 10 random numbers between 1 & 20
myrand=ceil(20*rand([1 10]));
%{
There are multiple inefficiencies in the original code.
One is the use of a for loop rather than a vectorised input, but another is
that we didn't initialise the myrand variable before the loop, which means
that it needs to change size on each iteration, leading to unnecessary data 
copying and consequent slowing. Let's look in more detail at how these
inefficiencies add up...
%}

n=100000;

% Optimised code:
clear myrand
tic
myrand=ceil(20*rand([1 n]));
s=toc;
fprintf('\n Optimised code took %g seconds',s)

% Using a for loop instead of vectorised input, 
% but prealocating the myrand variable:
clear myrand
tic
myrand=zeros(1,n);
for ind=1:n
    myrand(ind)=ceil(20*rand(1));
end;
s=toc;
fprintf('\n Looped code with preallocation took %g seconds',s)

% Using a for loop instead of vectorised input, 
% without prealocating the myrand variable:
clear myrand
tic
for ind=1:n
    myrand(ind)=ceil(20*rand(1));
end;
s=toc;
fprintf('\n Looped code without preallocation took %g seconds',s)

%% Find the square of the numbers from 1-10
mysquares=(1:10).^2;

%% Multiply each element in "a" by each element in "b"
a=[1 2 3 4];
b=[1 2 1 3];
c=a.*b;

%% Given the two dimensional matrix x, subtract the mean of each row
tic
x=[1 2 3 4; 2 3 4 5; 7 6 5 4; -1 -2 -3 -4];
xrownorm=x-repmat(mean(x,2),[1 size(x,2)]);
%toc
% or in newer versions of matlab: bsxfun(@minus,x,mean(x,2))

%% Find the index of the first occurence of "chicken"
x={'picasso','rembrandt','gaugin','chicken','monet'};
% chickenpos=find(strcmp(x,'chicken'),1); % this is one option
% chickenpos=strmatch('chicken',x); % this is even better 
chickenpos=find(strcmp('chicken',x)); % matlab recommends this:
% could also use regexp() for more complicated string matching 

if (chickenpos<=length(x))
    fprintf('\nThe first chicken is at position %d',chickenpos);
else
    fprintf('\nNot found.');
end;

%% Find the first artist to get 5 points or more
clear x
x(1).name='Picasso';
x(1).quality=4;
x(2).name='Monet';
x(2).quality=5;
x(3).name='Gaugin';
x(3).quality=3;
artistpos=find([x.quality]>=5,1);
fprintf('\nThe artist is %s\n',x(artistpos).name);

    