function demos_simpletoydata_234_djm()
% These are demos showing some simple decoding on simulated toy data.
% The toy data are simple matlab matrices and no "real" fMRI or EEG data.
%
% Modified by djm. 1/3/17, 5/9/22
% Based on demos 2-4 of The Decoding Toolbox

close all
dbstop if error % in case something goes wrong
set(0, 'DefaultAxesFontSize', 7);

%% add paths to required files
fpath = fileparts(which(mfilename)); % location of current file
addpath(fpath) % add this directory to path
addpath(fileparts(fileparts(fpath))) % for TDT
addpath(fileparts(fileparts(fileparts(fileparts(fpath))))); % for plotSVM_djm.m

%% initialize toolbox & create 'cfg' structure to hold all settings
cfg = decoding_defaults; % can examine cfg structure to see default settings

% Set the analysis style that should be performed (here we only want to do 1 decoding)
cfg.analysis = 'wholebrain'; % to use all voxels...
% ...for real data this could also be 'ROI' or 'searchlight'

%% set seed for random number generator (to get reproducible output)
rng(99);

%% "Demo 2": generate some toy data
% define number of "runs" and center means
nruns = 4; % let's simulate we have n=4 runs

class1.mean = [.2 .2]; % two voxels; means for class 1
class2.mean = [0 0]; % two voxels; means for class 2

% simulate data on two shifted lines
x = randn(nruns, 1)*.25; % a random number per run
dat = [x, -x]; % voxels will share negatively correlated noise across runs
data1 = dat + repmat(class1.mean, nruns, 1); % classes have same noise distribution and are separated by their mean offset in orthogonal direction
data2 = dat + repmat(class2.mean, nruns, 1);

% alternative:
% data1 = randn(nruns, length(class1.mean)) + repmat(class1.mean, nruns, 1);
% data2 = randn(nruns, length(class2.mean)) + repmat(class2.mean, nruns, 1);

% put all together in a data matrix
data = [data1; data2]; % samples x voxels

%% add data description
% save labels (each sample is from class 1 or 2)
cfg.files.label = [ones(nruns, 1); 2*ones(nruns, 1)];

% save run/chunk number of each sample
cfg.files.chunk = [1:nruns, 1:nruns]';

% save a description for each sample (see next example for more automatic filling)
for i = 1:length(cfg.files.label)
        cfg.files.name(i) = {sprintf('class%i_run%i', cfg.files.label(i), cfg.files.chunk(i))};
end

% add an empty mask
cfg.files.mask = '';

%% plot the (2d) data
resfig = 1;
figure(resfig); set(resfig,'name', 'Demo 2 - toy data, 2 voxels, 2 classes'); clf(resfig)
scatter(data(:, 1), data(:, 2), 80, cfg.files.label,'filled');
axis square

keyboard % pause to view data

%% Prepare data for passing (see next example for more automatic filling)
passed_data.data = data;
passed_data.mask_index = 1:size(data, 2); % use all voxels
passed_data.files = cfg.files;
passed_data.hdr = ''; % we don't need a header, because we don't write img-files as output (but mat-files)
passed_data.dim = [length(class1.mean), 1, 1]; % add dimension information of the original data
% passed_data.voxelsize = [1 1 1];

%% Set the output options
cfg.results.write = 0; % no results are written to disk in this example
% Here we won't save results, but we could set the output directory where data would be saved
% cfg.results.dir = % e.g. 'toyresults'

cfg.results.output = {'accuracy', 'model_parameters'}; % which outputs to calculate; add 'model_parameters' to see the fitted model (i.e. classifying hyperplane; used for plotting below)
cfg.plot_selected_voxels = 0; % progress report - e.g. see example in demos_toybrain_567_djm

%% Create, print & plot design
cfg.design = make_design_cv(cfg); % Create a leave-one-run-out cross validation design

display(display_design(cfg));

cfg.fighandles.plot_design = 2;
plot_design(cfg);
set(cfg.fighandles.plot_design,'name', 'Demo 2 - design');

keyboard % pause to view decoding design; return to slides to discuss CV and "decoding design"

%% Set decoding Parameters
cfg.decoding.method = 'classification';
cfg.decoding.train.classification.model_parameters = '-s 0 -t 0 -c 1 -b 0 -q';
%{
These settings are for the default LIBSVM classifiers - see
https://www.csie.ntu.edu.tw/~cjlin/libsvm/)

-s svm_type :
	0 -- C-SVC		(multi-class classification)
	1 -- nu-SVC		(multi-class classification)
	2 -- one-class SVM
	3 -- epsilon-SVR	(regression)
	4 -- nu-SVR		(regression)
-t kernel_type :
	0 -- linear: u'*v
	1 -- polynomial: (gamma*u'*v + coef0)^degree
	2 -- radial basis function: exp(-gamma*|u-v|^2)
	3 -- sigmoid: tanh(gamma*u'*v + coef0)
	4 -- precomputed kernel (kernel values in training_set_file)
-c cost : set the parameter C of C-SVC (and epsilon-SVR, and nu-SVR) (default 1)
-n nu : set the parameter nu of nu-SVC (and one-class SVM, and nu-SVR) (default 0.5)
-b probability_estimates : whether to train a SVC or SVR model for probability estimates
-q : quiet mode
%}

%% Disable scaling min0max1 to allow estimating model_parameters
% If you dont need model parameters, and if you use libsvm, we would use:
% cfg.scale.method = 'min0max1';
% cfg.scale.estimation = 'all'; % scaling across all data is equivalent to no scaling (i.e. will yield the same results), it only changes the data range which allows libsvm to compute faster
% But because we want to get the model parameters (to plot the classification boundary) we disable scaling
cfg.scale.method = 'none';
% and acknowledge that you know that libsvm can be very slow without scaling
cfg.scale.IKnowThatLibsvmCanBeSlowWithoutScaling = 1;

%% Run decoding!
[results, cfg, used_data] = decoding(cfg, passed_data);

%% Plot results and decision boundary
plotSVM_djm(results,cfg,used_data,[],resfig)

keyboard % pause to view decoding results

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% "Demo 3": now create some 3-class toy data
nruns = 8; % lets simulate we have n runs
class1.mean = [0 0];
class2.mean = [1 1]; % should have the same dim as set1, otherwise it wont work (and would not make sense, either)
class3.mean = [-1 1];

% uniform noise (leads to squares around the mean)
data1 = rand(nruns, length(class1.mean))-0.5 + repmat(class1.mean, nruns, 1);
data2 = rand(nruns, length(class2.mean))-0.5 + repmat(class2.mean, nruns, 1);
data3 = rand(nruns, length(class3.mean))-0.5 + repmat(class3.mean, nruns, 1);

% put all together in a data matrix
data = [data1; data2; data3];

%% describe data and prepare for passing

% make labels
label = [1*ones(size(data1,1), 1); 2*ones(size(data2,1), 1); 3*ones(size(data3,1), 1)];

% set "run" as chunk number
chunk = [1:nruns, 1:nruns, 1:nruns]';

% prepare data for passing
passed_data.data = data;
cfg.files=struct('label',{[]},'chunk',{[]},'name',{{}});
[passed_data,cfg] = fill_passed_data(passed_data,cfg,label,chunk);
% (labels and chunks added; txt description and other fields generated automatically)

%% plot the (2d) data
figure(resfig); set(resfig,'name', 'Demo 3 - toy data, 2 voxels, 3 classes'); clf(resfig)
scatter(data(:, 1), data(:, 2), 80, cfg.files.label,'filled');
axis square

keyboard % pause to view data

%% Create, print & plot design
cfg.design = make_design_cv(cfg); % Create a leave-one-run-out cross validation design
display(display_design(cfg));
plot_design(cfg);
set(cfg.fighandles.plot_design,'name', 'Demo 3 - design');

keyboard % pause to view design

%% Run decoding!
[results, cfg, used_data] = decoding(cfg, passed_data);

%% Plot results and decision boundary
plotSVM_djm(results,cfg,used_data,[],resfig)

keyboard % pause to view results

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% "Demo 4" now define some UNBALANCED toy data
% point clouds around two shifted lines

nruns = 8; % lets simulate we have n runs
class2.mean = [0 0];
class1.mean = [.2 .2]; % should have the same dim as set1, otherwise it wont work (and would not make sense, either)

class1.fact = 5; % unbalanced samples per run; i.e. more of class 1
class2.fact = 1;

label = [ones(class1.fact, 1); 2 * ones(class2.fact, 1)];
label = repmat(label, nruns, 1);

noise_covariance = .2; % 0: all along one line
signal_covariance = 0.2; % 0: all on one spot
r = pi/4; % rotation from 0 .. 2pi

%% generate data
clear data data1 data2
data1(:,1) = class1.mean(1) + randn(class1.fact * nruns, 1) * noise_covariance;
data1(:,2) = class1.mean(2) + randn(class1.fact * nruns, 1) * signal_covariance;
data(label == 1, :) = data1;

data2(:,1) = class2.mean(1) + randn(class2.fact * nruns, 1) * noise_covariance;
data2(:,2) = class2.mean(2) + randn(class2.fact * nruns, 1) * signal_covariance;
data(label == 2, :) = data2;

data = data * [cos(r) sin(r); -sin(r) cos(r)]; % rotate data

%% describe data and prepare for passing

% save "run" as chunk number
chunk = [];
for ri = 1:nruns
        curr_run = ri * ones(class1.fact + class2.fact ,1);
        chunk = [chunk; curr_run]; % save run number
end

% prepare data for passing
passed_data.data = data;
cfg.files=struct('label',{[]},'chunk',{[]},'name',{{}});
[passed_data,cfg] = fill_passed_data(passed_data,cfg,label,chunk);
% (labels and chunks added; txt description and other fields generated automatically)

%% plot the (2d) data
figure(resfig); set(resfig,'name', 'Demo 4 - toy data, 2 voxels, 2 unbalanced classes'); clf(resfig)
scatter(data(:, 1), data(:, 2), 80, cfg.files.label,'filled');
axis square

keyboard % pause to view data

%% Create, print & plot design
cfg.design = make_design_cv(cfg); % Create a leave-one-run-out cross validation design
display(display_design(cfg));
plot_design(cfg);
set(cfg.fighandles.plot_design,'name', 'Demo 4 - design');

keyboard % pause to view design

%% set some decoding options:
cfg.design.unbalanced_data = 'ok';
disp(['cfg.design.unbalanced_data = ' cfg.design.unbalanced_data]);
cfg.results.output = {'accuracy','balanced_accuracy','dprime','auc', 'model_parameters'}; % which outputs to calculate; this time we'll add a few more; add 'model_parameters' to see the fitted model (i.e. classifying hyperplane; used for plotting below)

%% first run decoding using equal weights
txt={'Using equal weights'};
weight_str = '';

cfg.decoding.train.classification.model_parameters = ['-s 0 -t 0 -c 1000 -b 0 ' weight_str  ' -q'];

[results, cfg, used_data] = decoding(cfg, passed_data);

for o=cfg.results.output
        switch o{1}
                case {'model_parameters'}
                otherwise
                        txt=[txt sprintf('%s = %.1f',o{1},results.(o{1}).output)];
        end
end

plotSVM_djm(results,cfg,used_data,[],resfig,txt)

keyboard % pause to view results

%% now rerun decoding using balanced weights

% if we use weights, these must be inverse to the amount of data
% e.g., if we use 5 times as much examples for class 1, we need
%   -w1 1 -w2 5
% so in the example here, we just use .fact of the other set
% '#' in '-w#' must correspond to each label value

txt={'Using correct (balanced) weights'};
weight_str = ['-w1 ' num2str(class2.fact) ' -w2 ' num2str(class1.fact)];

cfg.decoding.train.classification.model_parameters = ['-s 0 -t 0 -c 1000 -b 0 ' weight_str  ' -q'];

[results, cfg, used_data] = decoding(cfg, passed_data);

for o=cfg.results.output
        switch o{1}
                case {'model_parameters'}
                otherwise
                        txt=[txt sprintf('%s = %.1f',o{1},results.(o{1}).output)];
        end
end

plotSVM_djm(results,cfg,used_data,[],resfig+2,txt)

keyboard % pause to view results

%%%%%%% Q) Which has higher accuracy? Which is better? How else might we
%%%%%%% correct for bias? (cross-validation scheme? performance measure?)

return