function tutorial_djm2()
%{
MVPA tutorial based on summary data from Mitchell & Cusack (2016) "Semantic 
and emotional content of imagined representations in human occipitotemporal
cortex"
(Note, this script is for demonstration purposes only - the example data are 
combined across subjects which might invalidate the statistics.)


% LINEAR SVM:
%  - train the classifier
%       train a linear SVM classifier to distinguish animate from inanimate
%       objects
%  - test the classifier
%  - statistical inference
%       use a condition-label permutation test to determine whether the
%       classifier performs above chance

% RSA:
%  - generate an RDM for each session
%  - compare RDMs across sessions
%  - relate brain RDM to model RDMs

%}

close all
dbstop if error

workingdir = fileparts(which(mfilename));

analyse.lSVM=0;
analyse.RSA=1;

%% load software and data
addpath(fullfile(workingdir,'tdt_3.999E2','decoding_toolbox','decoding_software','libsvm3.17','matlab'));
addpath(genpath(fullfile(workingdir,'rsa-master')));
import rsa.*
import rsa.rdm.*
import rsa.fig.*
import rsa.stat.*
import rsa.sim.*
import rsa.fmri.*
import rsa.util.*
import rsa.spm.*

% fMRI patterns
load(fullfile(workingdir,'imageryexp','compromise.mat'),'compromise') % 'compromise' is a cell array of the cross-subject compromise patterns (objects x "voxels") % (actually the voxels are principle components)
sessions=[2 4]; % 2 and 4 are the two imagery conditions
patterns=cellfun(@(x) x.dat',compromise(sessions),'uniformoutput',0); %#ok<USENS> % now voxel x object

% stimulus features
load(fullfile(workingdir,'imageryexp','features.mat'),'features') %'features' contains the score per object for various sensory/semantic/emotional features

% stimulus images
load(fullfile(workingdir,'imageryexp','imgs.mat'),'imgs'); % 'imgs' is cell array of thumbnails per object
[imageIcons(1:length(imgs)).image]=deal(imgs{:}); %#ok<IDISVAR,USENS>

% sort them all by animacy
[~, ind]=sort(features.SUB_Animacy); %#ok<NODEF>
features=structfun(@(x) x(ind),features,'uniformoutput',0);
imageIcons=imageIcons(ind);
patterns=cellfun(@(x) x(:,ind),patterns,'uniformoutput',0);

% define labels based on animacy
%labels=double(features.SUB_Emotional_V(:)>median(features.SUB_Emotional_V));
labels=double(features.SUB_Animacy>0)+1;
labels=labels(:);

%% animate vs. inanimate classification using linear SVM

if analyse.lSVM % (just need 'patterns' and 'labels')
        
        % options for support vector machine
        libSVMsettings='-s 1 -t 0 -q'; % nu-SVM, linear
        % edit libsvm_README.txt
        % low c (~>0) or hi n (~<1) allow more SVs (less penalty for training
        % errors, wider margin)
        
        nRandomisations=1000;
        
        % linear SVM
        cvFolds=[1 2; 2 1]; % columns = folds, row 1 = session used for training, row 2 = session used for testing.
        % i.e. first train on session 1 and test on session 2, then the opposite.
        accuracy_fold=nan(1,2);
        accuracy_randomisation_fold=nan(nRandomisations,2);
        for foldI=1:size(cvFolds,2)
                fprintf('\nSVM fold %d of %d.',foldI,2)
                % define training and test data sets
                patternsTrain=patterns{cvFolds(1,foldI)}; patternsTrain=double(patternsTrain');
                patternsTest =patterns{cvFolds(2,foldI)}; patternsTest =double(patternsTest');
                
                % train and test the classifier
                model=svmtrain(labels,patternsTrain,libSVMsettings); %#ok<SVMTRAIN>
                [labelsPredicted, accuracy, dv]=svmpredict(labels,patternsTest,model,'-q');
                % accuracy(1) is percent correct:                 mean(labels==labelsPredicted) * 100
                % accuracy(2) is mean squared error:              mean((labels-labelsPredicted).^2)
                % accuracy(3) is squared correlation coefficient: corr(labels,labelsPredicted).^2
                % Third output would be a matrix containing decision values (or probability
                % estimates if '-b 1' is specified for both training and testing).
                
                accuracy_fold(foldI)=accuracy(1);
                
                %%% visualise
                % (in this case train and test labels are the same)
                figure(999); clf
                plotSVM_djm(patternsTrain,patternsTest,labels,labels,model)
                %%%
                
                % create null distribution for statistical inference
                fprintf(' Creating null distribution from %d label permutations',nRandomisations)
                tic
                for randomisationI=1:nRandomisations
                        % randomise labels
                        labelsRand=labels(randperm(length(labels)));
                        
                        % train and test the classifier using the randomised training labels
                        % modelRand=svmtrain(labelsRand,patternsTrain,libSVMsettings);
                        %[labelsPredictedRand,accuracyRand,dv]=svmpredict(labels,patternsTest,modelRand,'-q');
                        
                        % or randomised testing labels?
                        %[labelsPredictedRand,accuracyRand,dv]=svmpredict(labelsRand,patternsTest,model,'-q');
                        
                        % or both
                        modelRand=svmtrain(labelsRand,patternsTrain,libSVMsettings); %#ok<SVMTRAIN>
                        [labelsPredictedRand,accuracyRand,dv]=svmpredict(labelsRand,patternsTest,modelRand,'-q');
                        
                        accuracy_randomisation_fold(randomisationI,foldI)=accuracyRand(1);
                        
                        if ~mod(randomisationI,100), fprintf('.'); end
                end % randomisationI
                toc
        end % foldI
        
        % statistical inference
        accuracy=mean(accuracy_fold);
        accuracyH0=mean(accuracy_randomisation_fold,2);
        p=1-relRankIn_includeValue_lowerBound(accuracyH0,accuracy); % proportion of null distribution smaller than true value
        
        % visualise results
        figure(3); clf;
        set(3,'Color','w','name','SVM classification accuracy with perputation test')
        % plot null distribution
        hist(accuracyH0,25); hold on;
        % plot accuracy (mean across folds) found in the data
        plot(accuracy,0,'o','MarkerEdgeColor','r','MarkerFaceColor','r','MarkerSize',8);
        axis tight
        xlim([0 100])
        box off
        ylims=ylim;
        % add labels
        text(accuracy,0.06*ylims(2),{'\bfdata ',['p = ',sprintf('%1.3f',p)]},'Color','r');
        xlabel('accuracy');
        ylabel('frequency');
        title({'\fontsize{11}null distribution of classification accuracy',['\fontsize{8}',num2str(nRandomisations),' stimulus-label randomisations']})
        
        keyboard
end % do classification?


%% RSA

if analyse.RSA
        
        % make an RDM per session
        RDMs=struct();
        for i=1:2
                % compute the square RDMs using the pdist function
                RDMs(1,i).RDM   = squareform(pdist(patterns{i}','correlation'));
                RDMs(1,i).name  = sprintf('hIT RDM | %s | session %d',compromise{sessions(i)}.name,i); %#ok<IDISVAR>
                RDMs(1,i).color = [0 0 0];
        end
        
        % show the 2 session RDMs (note conditions have been sorted by animacy)
        figI=4;
        figure(figI);clf;set(figI,'Position',[100 100 800 800],'Color','w')
        showRDMs(RDMs,figI);
        
        % compare RDMs across sessions
        [r12, p12]=corr(vectorizeRDMs(RDMs(1).RDM)',vectorizeRDMs(RDMs(2).RDM)');
        subplot(2,2,4)
        plot(vectorizeRDMs(RDMs(1).RDM),vectorizeRDMs(RDMs(2).RDM),'.')
        xlabel('Dissimilarities from session 1')
        ylabel('Dissimilarities from session 2')
        axis tight; refline
        box off
        title(sprintf('r = %.3f, (p=%.3f, invalid due to dependence)',r12,p12))
        % Note the p-value is not valid due to non-independence of
        % dissimilarities)
        
        % combine the two sessions
        avgRDM = averageRDMs_subjectSession(RDMs,'subject');
        avgRDM.name='hIT RDM averaged across sessions';
        figI=5;
        figure(figI);set(figI,'Position',[100 100 800 800],'Color','w')
        showRDMs(avgRDM,figI);
        
        % do multidimensional scaling
        options.MDScriterion='stress';
        [pats_mds_2D,stress,disparities]=mdscale(avgRDM.RDM,2,'criterion',options.MDScriterion);
        
        % compute image size
        imageAreaProportion=.5;
        boundingBoxArea=max(prod(range(pats_mds_2D)),max(range(pats_mds_2D))^2/10);
        totalImageArea=boundingBoxArea*imageAreaProportion;
        imageWidth=sqrt(totalImageArea/length(avgRDM.RDM));
        
        figI=6;
        figure(figI);clf;
        set(figI,'Position',[ 100 100 800 800],'Color',[1 1 1],'Renderer','OpenGL','BackingStore','on');
        axes('Position',[0.05 0.2 0.9 0.75])
        hold on
        for im=1:length(avgRDM.RDM)
                image('CData',imageIcons(im).image,'XData',pats_mds_2D(im,1) + [-imageWidth/2, +imageWidth/2],'YData',pats_mds_2D(im,2) + [imageWidth/2 -imageWidth/2],'AlphaData',0.6);
        end
        axis tight equal off
        
        % add a micro mds plot
        axes('Position',[0.7 0 0.25 0.25])
        hold on;
        % choose some colours for the different stimuli, based on their features
        cmap=jet(200);
        options.categoryColors=cmap(round(tiedrank(features.SUB_Animacy)),:);
        
        markerSize=6;
        for im=1:length(avgRDM.RDM)
                
                plot(pats_mds_2D(im,1),pats_mds_2D(im,2),...
                        'o','MarkerFaceColor',options.categoryColors(im,:),'MarkerEdgeColor',options.categoryColors(im,:),'MarkerSize',markerSize);
        end
        axis tight equal off
        
        % relate hIT and models
        featurenames=fieldnames(features);
        for m=1:length(featurenames)
                modelRDMs{m}.RDM=squareform(pdist(features.(featurenames{m})(:)));
                modelRDMs{m}.name=featurenames{m};
                modelRDMs{m}.color=[1 0 0];
        end
        
        showRDMs([modelRDMs{:}])
        
        userOptions.RDMrelatednessTest='randomisation'; % fixed effects
        userOptions.candRDMdifferencesTest='conditionRFXbootstrap'; % 'none' crashes, because wants to add lines??
        userOptions.candRDMdifferencesMultipleTesting='FDR'; % case sensitive
        userOptions.nRandomisations=1000;
        userOptions.nBootstrap=200; % default 10000, not 1000 as stated in help? % 200 is too few, but still takes ~15 minutes
        userOptions.RDMcorrelationType='Spearman'; % 'Kendall_taua' recommended, but slow and approximate
        userOptions.rootPath=workingdir;
        userOptions.saveFiguresPS=0; % defaults not specified correctly, so crashes if not prespecified?
        userOptions.saveFiguresPDF=0;
        userOptions.saveFiguresFig=0;
        userOptions.analysisName='demo';
        
        % visually compare ROIs and models:
        fig=7;
        allRDMs=[RDMs(:); vertcat(modelRDMs{:})]';
        RDMRDM=1-RDMCorrMat(allRDMs);
        userOptions.criterion='stress';
        figureMDSArrangement(RDMRDM, userOptions,...
                struct('titleString','MDS of different RDMs','fileName','2ndOrderMDSplot',...
                'textLabels',{{allRDMs.name}},'dotColours',vertcat(allRDMs.color),...
                'figI_textLabels',[fig 1 2 1],'figI_shepardPlots',[fig 1 2 2],...
                'rubberbandGraphPlot',true));
        
        % stats
        userOptions.figureIndex = [8 9]; figure(8); figure(9); set([8 9],'WindowState','maximized');
        t0=tic;
        stats_p_r=compareRefRDM2candRDMs(RDMs, modelRDMs, userOptions);
        fprintf('Took %.1f mins',toc(t0)/60)
        
        keyboard
end % do RSA?

return
