function DEMO5_Recipe_fMRI_djm()
% This function is based on a combination of
% Recipe_fMRI.m and Recipe_fMRI_searchlight.m
% using the SPM demo data from The Decoding Toolbox.
% Just one subject and 6 conditions, so no stats attempted.
% djm 4/2017, 5/9/22

%%%%%%%%%%%%%%%%%%%%
%% Initialisation %%
%%%%%%%%%%%%%%%%%%%%

close all
dbstop if error

% add paths to toolbox:
here=fileparts(which(mfilename));
addpath(genpath(fileparts(here)));
import rsa.*
import rsa.rdm.*
import rsa.fig.*
import rsa.stat.*
import rsa.sim.*
import rsa.fmri.*
import rsa.util.*
import rsa.spm.*

% set the output directory:
userOptions.rootPath = fullfile(here,'DEMO5');
userOptions.analysisName = 'DEMO5';
if ~exist(userOptions.rootPath,'dir'), mkdir(userOptions.rootPath); end

% details of data:

% Should information about the experimental design be automatically acquired from SPM metadata?
% If this option is set to true, the entries in userOptions.conditionLabels MUST correspond to the names of the conditions as specified in SPM.
userOptions.getSPMData = true;
userOptions.conditionLabels={'up','down','red','green','button left','button right'};
nCond=length(userOptions.conditionLabels);

% The path leading to where the scans are stored (not including subject-specific identifiers).
% "[[subjectName]]" should be used as a placeholder to denote an entry in userOptions.subjectNames
% "[[betaIdentifier]]" should be used as a placeholder to denote an output of betaCorrespondence.m if SPM is not being used; or an arbitrary filename if SPM is being used.
demodatadir=fullfile(here,'..','..','tdt_3.999E2','decoding_toolbox','demos','TDTdemo8data');
if ~exist(demodatadir,'dir')
        demodatadir=spm_select(1,'dir','Please locate folder TDT demo dataset',{},fileparts(here));
end
try
        copyfile(fullfile(demodatadir,'sub01_GLM','mask.*'),fullfile(demodatadir,'sub01_ROI'));
catch
end

userOptions.betaPath = fullfile(demodatadir,'[[subjectName]]_firstlevel','[[subjectName]]_GLM','[[betaIdentifier]]');
userOptions.maskPath = fullfile(demodatadir,'[[subjectName]]_firstlevel','[[subjectName]]_ROI','[[maskName]].img');
userOptions.maskNames = {'mask','m1_left','v1','v4_both','mt_both'};

userOptions.structuralsPath = fullfile(demodatadir,'[[subjectName]]_firstlevel','[[subjectName]]_struct');
userOptions=rmfield(userOptions,'structuralsPath'); % this is used for normalisation; remove the field if we don't want to do this!

userOptions.voxelSize = [2 2 3]; % dimensions (in mm) of the voxels in the scans
userOptions.searchlightRadius = 15; % radius of searchlight (mm)

userOptions.subjectNames = {'sub01'}; % The list of subjects to be included in the study.

% visualisation options:
userOptions.RoIColor = [0 0 1]; % colour for ROI RDMs
userOptions.ModelColor = [0 1 0]; % colour for model RDMs
userOptions.conditionColours = kron(hsv(3),[1;1]); % colours for conditions (6 conditions in 3 pairs)
userOptions.rankTransform = true; % Should RDMs' entries be rank transformed into [0,1] before they're displayed?
userOptions.criterion = 'metricstress'; % What criterion shoud be minimised in MDS display?
userOptions.rubberbands = true; % Should rubber bands be shown on the MDS plot?
userOptions.colourScheme = RDMcolormap(); % What is the colourscheme for the RDMs?
userOptions.plotpValues = '=';

% (dis)similarity measures:
userOptions.distance = 'Correlation'; % distance measure to calculate first-order RDMs.
userOptions.distanceMeasure = 'Spearman'; % similarity-measure used for the second-order comparison
userOptions.RDMcorrelationType=userOptions.distanceMeasure; % different names for same thing, depending on function??

% stats options: (we won't actually need these for this demo)
userOptions.significanceTestPermutations = 1000; % (10,000 recommended); this is overridden by .nRandomisations??
userOptions.nResamplings = 1000; % for bootstrapping
userOptions.resampleSubjects = true; % for bootstrapping
userOptions.resampleConditions = false; % for bootstrapping
userOptions.RDMrelatednessTest = 'none'; % because single subject and <8 conditions
userOptions.RDMrelatednessThreshold = 0.05;
userOptions.RDMrelatednessMultipleTesting = 'FWE';
userOptions.candRDMdifferencesTest = 'none'; % single subject and <20 conditions in this demo?
userOptions.candRDMdifferencesThreshold = 0.05;
userOptions.candRDMdifferencesMultipleTesting = 'none';

% How should figures be outputted?
userOptions.displayFigures = true;
userOptions.saveFiguresPDF = false;
userOptions.saveFiguresFig = false;
userOptions.saveFiguresPS = false;
userOptions.dpi = 300; % Which dots per inch resolution do we output?
userOptions.tightInset = false; % Remove whitespace from PDF/PS files?

userOptions.forcePromptReply = 'r';

%%%%%%%%%%%%%%%%%%%%%%
%% Data preparation %%
%%%%%%%%%%%%%%%%%%%%%%

fullBrainVols = fMRIDataPreparation('SPM', userOptions); % .subject([vox x cond x session])
binaryMasks_nS = fMRIMaskPreparation(userOptions); % .subject.mask([x y z])


%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Model RDM definition %%
%%%%%%%%%%%%%%%%%%%%%%%%%%

clear mymodelRDMs;
mymodelRDMs.BetweenModality=double(categoricalRDM([0 0 1 1 2 2],[],0)); % this creates the RDM
% a simple categorical model to test for a difference between sets of similar regressors
models = constructModelRDMs(mymodelRDMs, userOptions); % this 'wraps' it


%%%%%%%%%%%%%%%%%
%% Searchlight %%
%%%%%%%%%%%%%%%%%

% NOTE BUGFIX IN fMRISearchlight.m FOR MISSING FUNCTION spacesToUnderscores.m
% (also a bug with returned arguments)

tempoptions=userOptions;
tempoptions.maskNames={'mask'}; % just the wholebain mask!

%%%%% by default averages RDMs across sessions:
% fMRISearchlight(fullBrainVolsAvg, binaryMasks_nS, models, 'SPM', tempoptions);
%%%%% to speed things up we'll average betas over sessions instead:
% (expect about 1.5 minutes...)
fullBrainVolsAvg = structfun(@(x) mean(x,3),fullBrainVols,'uniformoutput',0);
betaids=getDataFromSPM(userOptions);
t0=tic;
fMRISearchlight(fullBrainVolsAvg, binaryMasks_nS, models, betaids(1,:), tempoptions); % (this calls searchlightMapping_fMRI internally)
% djm: I think it ignores the choice of userOptions.distance &
% userOptions.distanceMeasure!! Pearson correlation always used for 
% constructing RDMs and Spearman for comparing to model.
% Note also, that the output map is prefixed 'r', but it is actually Fisher transformed. 
fprintf('Took %.1f mins',toc(t0)/60)
%%%%%

% Now view .../Maps/...nii.

%%%%%%%%%%%%%%%%%%%%%%%%%
%% ROI RDM calculation %%
%%%%%%%%%%%%%%%%%%%%%%%%%

userOptions.maskNames={'m1_left','v1','v4_both','mt_both'};
responsePatterns = fMRIDataMasking(fullBrainVols, binaryMasks_nS, 'SPM', userOptions); % .mask.subject([vox x cond x session])
ssRDMs = constructRDMs(responsePatterns, 'SPM', userOptions); % [mask x subject x session]
sRDMs = averageRDMs_subjectSession(ssRDMs, 'session'); % [mask x subject]


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% First-order visualisation %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% variability across sessions for one ROI:
figureRDMs(ssRDMs(1,:,:), userOptions, struct('fileName', 'RoIRDMs', 'figureNumber', 1));

% variability across ROIs for average across sessions:
figureRDMs(sRDMs, userOptions, struct('fileName', 'RoIRDMs', 'figureNumber', 2));

% MDS for one ROI:
MDSConditions(sRDMs(4), userOptions);

% dendrogram for same ROI:
dendrogramConditions(sRDMs(4), userOptions);
% djm: There seems to be a bug in the assignment of labels to conditions??

%% add a few more models and display them:
mymodelRDMs.BetweenModalityPlusVisual=mymodelRDMs.BetweenModality;
mymodelRDMs.BetweenModalityPlusVisual(1,2)=0.5;
mymodelRDMs.BetweenModalityPlusVisual(2,1)=0.5;
mymodelRDMs.BetweenModalityPlusVisual(3,4)=0.5;
mymodelRDMs.BetweenModalityPlusVisual(4,3)=0.5;

mymodelRDMs.MotorVsVisual=double(categoricalRDM([0 0 0 0 1 1],[],0));

models = constructModelRDMs(mymodelRDMs, userOptions);

userOptions.rankTransform=false;
figureRDMs(models, userOptions, struct('fileName', 'ModelRDMs', 'figureNumber', 3));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% relationship amongst multiple RDMs %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

allRDMs=[sRDMs(:); models(:)]';

% compare ROIs and models with tau-a:
fig=4;
userOptions.distanceMeasure='Kendall_taua'; %
userOptions.RDMcorrelationType=userOptions.distanceMeasure; % different names for same thing, depending on function??

RDMRDM=1-RDMCorrMat(allRDMs,fig,userOptions.RDMcorrelationType);
showRDMs(1-RDMRDM,[fig 1 3 1],false,[],false); % no rank transform; no separate axis colourbar
colorbar
colormap(colorScale([0 0 1; .5 .5 .5; 1 0 0],256));
for RDMI=1:length(allRDMs)
        text(RDMI,RDMI,allRDMs(RDMI).name,'HorizontalAlignment','center','FontWeight','bold','interpreter','none','Color',allRDMs(RDMI).color,'FontUnits','normalized','FontSize',1/(length(allRDMs)*2));
end
title(['RDM correlation matrix (',userOptions.RDMcorrelationType,')'],'FontWeight','bold','interpreter','none');

figureMDSArrangement(RDMRDM, userOptions,...
        struct('titleString','MDS of different RDMs','fileName','2ndOrderMDSplot',...
        'textLabels',{{allRDMs.name}},'dotColours',vertcat(allRDMs.color),...
        'figI_textLabels',[fig 1 3 2],'figI_shepardPlots',[fig 1 3 3],...
        'rubberbandGraphPlot',true));

% compare ROIs and models with Spearman:
fig=5;
userOptions.distanceMeasure='Spearman'; %
userOptions.RDMcorrelationType=userOptions.distanceMeasure; % different names for same thing, depending on function??

RDMRDM=1-RDMCorrMat(allRDMs,fig,userOptions.RDMcorrelationType);
showRDMs(1-RDMRDM,[fig 1 3 1],false,[],false); % no rank transform; no separate axis colourbar
colorbar
colormap(colorScale([0 0 1; .5 .5 .5; 1 0 0],256));
for RDMI=1:length(allRDMs)
        text(RDMI,RDMI,allRDMs(RDMI).name,'HorizontalAlignment','center','FontWeight','bold','interpreter','none','Color',allRDMs(RDMI).color,'FontUnits','normalized','FontSize',1/(length(allRDMs)*2));
end
title(['RDM correlation matrix (',userOptions.RDMcorrelationType,')'],'FontWeight','bold','interpreter','none');

figureMDSArrangement(RDMRDM, userOptions,...
        struct('titleString','MDS of different RDMs','fileName','2ndOrderMDSplot',...
        'textLabels',{{allRDMs.name}},'dotColours',vertcat(allRDMs.color),...
        'figI_textLabels',[fig 1 3 2],'figI_shepardPlots',[fig 1 3 3],...
        'rubberbandGraphPlot',true));


%% compare RDMs over Sessions

allRDMs=squeeze(ssRDMs(2,:,:))';

fig=10;
userOptions.RDMcorrelationType=userOptions.distanceMeasure; % different names for same thing, depending on function??

RDMRDM=1-RDMCorrMat(allRDMs,fig,userOptions.RDMcorrelationType);
showRDMs(RDMRDM,[fig 1 3 1],false,[],false); % no rank transform; no separate axis colourbar
colorbar
colormap(colorScale([0 0 1; .5 .5 .5; 1 0 0],256));
for RDMI=1:length(allRDMs)
        text(RDMI,RDMI,allRDMs(RDMI).name,'HorizontalAlignment','center','FontWeight','bold','interpreter','none','Color',allRDMs(RDMI).color,'FontUnits','normalized','FontSize',1/(length(allRDMs)*2));
end
title(['RDM correlation matrix (',userOptions.RDMcorrelationType,')'],'FontWeight','bold','interpreter','none');

figureMDSArrangement(RDMRDM, userOptions,...
        struct('titleString','MDS of different RDMs','fileName','2ndOrderMDSplot',...
        'textLabels',{{allRDMs.name}},'dotColours',vertcat(allRDMs.color),...
        'figI_textLabels',[fig 1 3 2],'figI_shepardPlots',[fig 1 3 3],...
        'rubberbandGraphPlot',true));

% Observe: sessions closer in time have more similar representational
% structure

keyboard

return