function DEMO2_RSA_ROI_sim_djm()
% Sample recipe which will simulate some fMRI data and then run through an
% "RoI-based" RSA pipeline for the simulated data
%
% Cai Wingfield 5-2010, 6-2010, 7-2010
%__________________________________________________________________________
% Copyright (C) 2010 Medical Research Council
%
% modified by djm, 4/2017, 5/9/22

close all
dbstop if error

%%%%%%%%%%%%%%%%%%%%
%% Initialisation %%
%%%%%%%%%%%%%%%%%%%%

% add paths to toolbox:
here=fileparts(which(mfilename));
addpath(genpath(fileparts(here)));
import rsa.*
import rsa.rdm.*
import rsa.fig.*
import rsa.stat.*
import rsa.sim.*
import rsa.fmri.*

% set the output directory:
userOptions.rootPath = fullfile(here,'DEMO2');
userOptions.analysisName = 'DEMO2';
if ~exist(userOptions.rootPath,'dir'), mkdir(userOptions.rootPath); end

% Generate/populate the userOptions structure and later clone it for the two
% streams of data in this pipeline, changing only the analysis names.
% djm: I'll do it explicitly for this example, rather than calling a separate function.
% userOptions = projectOptions_demo();

% The path leading to where the scans are stored (not including subject-specific identifiers).
% "[[subjectName]]" should be used as a placeholder to denote an entry in userOptions.subjectNames
% "[[betaIdentifier]]" should be used as a placeholder to denote an output of betaCorrespondence.m if SPM is not being used; or an arbitrary filename if SPM is being used.
userOptions.betaPath = fullfile(here,'demoTest','[[subjectName]]','[[betaIdentifier]]');

% The list of subjects to be included in the study.
nsub=12;
userOptions.subjectNames = cellfun(@(x) sprintf('subject%d',x),num2cell(1:nsub),'uniformoutput',0);

% The default colour labels for RDMs corresponding to RoI masks or models.
userOptions.RoIColor = [0 0 1];
userOptions.ModelColor = [0 1 0];

% Text labels which may be attached to the conditions for MDS plots.
ncond=64;
userOptions.conditionLabels = cellfun(@(x) sprintf('condition%d',x),num2cell(1:ncond),'uniformoutput',0);
[userOptions.alternativeConditionLabels{1:ncond}] = deal(' ');
userOptions.useAlternativeConditionLabels = true;

% Colours for the conditions.
userOptions.conditionColours = kron([1 0 0; 0 1 0; 0 0 1; 1 .7 0], ones(16,1)); % here the 64 conditions are allocated to 4 groups

% Groups of conditions to plot convex hulls around
userOptions.convexHulls = [ones(1,16) 2*ones(1,16) 3*ones(1,16) 4*ones(1,16)]; % here the 64 conditions are allocated to 4 groups

% Which distance measure to use when calculating first-order RDMs.
userOptions.distance = 'Correlation';

% Which similarity-measure is used for the second-order comparison.
userOptions.distanceMeasure = 'Spearman';
userOptions.RDMcorrelationType=userOptions.distanceMeasure; % different names for same thing, depending on function??

% How many permutations should be used to test the significance of the fits?
userOptions.significanceTestPermutations = 1000; % (10,000 highly recommended); this is overridden by .nRandomisations??

% Should RDMs' entries be rank transformed into [0,1] before they're displayed?
userOptions.rankTransform = true;

% RDM Colourscheme
userOptions.colourScheme = jet(64);

% Should rubber bands be shown on the MDS plot?
userOptions.rubberbands = true;

% What criterion shoud be minimised in MDS display?
userOptions.criterion = 'metricstress';

% How should figures be outputted?
userOptions.displayFigures = true;
userOptions.saveFiguresPDF = false;
userOptions.saveFiguresFig = false;
userOptions.saveFiguresPS = false;
userOptions.saveFiguresEps = false;

% Bootstrap options
userOptions.nResamplings = 1000;
userOptions.resampleSubjects = true;
userOptions.resampleConditions = true;

userOptions.plottingStyle = 2; % ?

% Generate a simulationOptions structure.
simulationOptions = simulationOptions_demo();

%%%%%%%%%%%%%%%%
%% Simulation %%
%%%%%%%%%%%%%%%%
% This section generates simulated data and the details are not too important.
% The 'brain' will be a 7x7x7 cube
% Condition patterns will form 4 hierachical clusters

% Generate the SPM files for each subject containing conditions clustered
% according to preferences in the simulationOptions.
promptOptions.checkFiles(1).address = fullfile(userOptions.rootPath, 'Details', [userOptions.analysisName, '_simulateDataFiles_Details.mat']);
promptOptions.checkFiles(2).address = fullfile(userOptions.rootPath, 'Details', [userOptions.analysisName, 'True_fMRIDataPreparation_Details.mat']);
promptOptions.checkFiles(3).address = fullfile(userOptions.rootPath, 'Details', [userOptions.analysisName, 'Noisy_fMRIDataPreparation_Details.mat']);

for fileCount = 1:numel(promptOptions.checkFiles)
        if exist(promptOptions.checkFiles(fileCount).address, 'file')
                promptFlag(fileCount) = true; % If any of the passed-in files already exist, this will get toggled to true
        end%if
end%for
if ~exist('promptFlag','var'), promptFlag = false;end
if prod(double(promptFlag))
        reply = input('You have done the simulations before.Do you want to run it again? Y/N [Y]: ', 's');
        if ismember(reply,{'n','N','NO','No','no'})
                userOptions.forcePromptReply = 'S';
        else
                userOptions.forcePromptReply = 'R';
        end
end
userOptions_true = userOptions; userOptions_true.analysisName = [userOptions_true.analysisName 'True'];
userOptions_noisy = userOptions; userOptions_noisy.analysisName = [userOptions_noisy.analysisName 'Noisy'];
[betaCorrespondence_true,betaCorrespondence_noisy,fMRI] = simulateDataFiles(userOptions, simulationOptions);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Load data (patterns per condition per session per subject) %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Load in the 'true' fMRI data
fullBrainVols_true = fMRIDataPreparation(betaCorrespondence_true, userOptions_true);
% This structure has one field per subject, containing [voxels x conditions]

% Load in the 'noisy' fMRI data
fullBrainVols_noisy = fMRIDataPreparation(betaCorrespondence_noisy, userOptions_noisy);

% Name the RoIs for both streams of data
RoIName = 'SimRoI';
responsePatterns_true.(['true' RoIName]) = fullBrainVols_true;
responsePatterns_noisy.(['noisy' RoIName]) = fullBrainVols_noisy;

%%%%%%%%%%
%% RDMs %%
%%%%%%%%%%

% Construct RDMs for the 'true' data. One RDM for each subject (sessions
% have not been simulated) and one for the average across subjects.
RDMs_true = constructRDMs(responsePatterns_true, betaCorrespondence_true, userOptions_true);
averageRDMs_true = averageRDMs_subjectSession(RDMs_true, 'subject');

% Do the same for the 'noisy' data.
RDMs_noisy = constructRDMs(responsePatterns_noisy, betaCorrespondence_noisy, userOptions_noisy);
averageRDMs_noisy = averageRDMs_subjectSession(RDMs_noisy, 'subject');

% Prepare the model RDMs.
RDMs_model = constructModelRDMs(modelRDMs_demo2, userOptions); % This loads 4 very simple categorical models, plus a random model

%%%%%%%%%%%%%%%%%%%%%%%%%%
%% First-order analysis %%
%%%%%%%%%%%%%%%%%%%%%%%%%%

% % Display the three sets of RDMs: true, noisy and model
figureRDMs(concatenateRDMs(RDMs_true, averageRDMs_true), userOptions_true, struct('fileName', 'noiselessRDMs', 'figureNumber', 1));
set(1,'name','noiseless RDMs')

figureRDMs(concatenateRDMs(RDMs_noisy, averageRDMs_noisy), userOptions_noisy, struct('fileName', 'noisyRDMs', 'figureNumber', 2));
set(2,'name','noisy RDMs')

figureRDMs(RDMs_model, userOptions, struct('fileName', 'modelRDMs', 'figureNumber', 3));
set(3,'name','model RDMs')

% Display dendrograms for the clustering of the conditions for the two data streams
dendrogramConditions(averageRDMs_true, userOptions_true, struct('titleString', 'Dendrogram of conditions without simulated noise', 'useAlternativeConditionLabels', true, 'alternativeConditionLabels', {userOptions.alternativeConditionLabels}, 'figureNumber', 4));
set(4,'name','Dendrogram of conditions without simulated noise')
dendrogramConditions(averageRDMs_noisy, userOptions_noisy, struct('titleString', 'Dendrogram of conditions with simulated noise', 'useAlternativeConditionLabels', true, 'alternativeConditionLabels', {userOptions.alternativeConditionLabels}, 'figureNumber', 5));
set(5,'name','Dendrogram of conditions with simulated noise')

% Display MDS plots for the condition sets for both streams of data
MDSConditions(averageRDMs_true, userOptions_true, struct('titleString', 'MDS of conditions without simulated noise', 'alternativeConditionLabels', {userOptions.alternativeConditionLabels}, 'figureNumber', 6));
MDSConditions(averageRDMs_noisy, userOptions_noisy, struct('titleString', 'MDS of conditions with simulated noise', 'alternativeConditionLabels', {userOptions.alternativeConditionLabels}, 'figureNumber', 7));
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Second-order analysis %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Display a second-order simmilarity matrix for the models and the true and noisy simulated pattern RDMs
pairwiseCorrelateRDMs({averageRDMs_true, averageRDMs_noisy, RDMs_model}, userOptions, struct('figureNumber', 8));

% Plot all RDMs on a MDS plot to visualise pairwise distances.
MDSRDMs({averageRDMs_true, averageRDMs_noisy, RDMs_model}, userOptions, struct('titleString', 'MDS of noisy RDMs and models', 'figureNumber', 9));

%% statistical inference:
% test the relatedness and compare the candidate RDMs

userOptions = userOptions_noisy;
userOptions.RDMcorrelationType='Kendall_taua';
userOptions.RDMrelatednessTest = 'subjectRFXsignedRank';
userOptions.RDMrelatednessThreshold = 0.05;
userOptions.figureIndex = [10 11];
userOptions.RDMrelatednessMultipleTesting = 'FDR';
userOptions.candRDMdifferencesTest = 'subjectRFXsignedRank';
userOptions.candRDMdifferencesThreshold = 0.05;
userOptions.candRDMdifferencesMultipleTesting = 'none';
stats_p_r=compareRefRDM2candRDMs(RDMs_noisy, num2cell([RDMs_model averageRDMs_true]), userOptions);

return
