function DEMO4_RSAsearchlight_sim_djm()
%% DEMO4_RSAsearchlight_sim
% simulates fMRI data for a number of subjects and runs searchlight
% analysis using RSA, computes the similarity maps for each subject
% and does group level inference.
%
% modified by djm, 4/2017, 5/9/22

close all
dbstop if error

%%%%%%%%%%%%%%%%%%%%
%% Initialisation %%
%%%%%%%%%%%%%%%%%%%%

% add paths to toolbox:
here=fileparts(which(mfilename));
addpath(genpath(fileparts(here)));
import rsa.*
import rsa.rdm.*
import rsa.fig.*
import rsa.stat.*
import rsa.sim.*
import rsa.fmri.*
import rsa.util.*

% set the output directory:
userOptions.rootPath = fullfile(here,'DEMO4');
userOptions.analysisName = 'DEMO4';
if ~exist(userOptions.rootPath,'dir'), mkdir(userOptions.rootPath); end

% Generate/populate the userOptions structure.
% djm: I'll do it explicitly here, rather than calling a separate function.
% userOptions = projectOptions_demo();

% The path leading to where the scans are stored (not including subject-specific identifiers).
% "[[subjectName]]" should be used as a placeholder to denote an entry in userOptions.subjectNames
% "[[betaIdentifier]]" should be used as a placeholder to denote an output of betaCorrespondence.m if SPM is not being used; or an arbitrary filename if SPM is being used.
userOptions.betaPath = fullfile(here,'demoTest','[[subjectName]]','[[betaIdentifier]]');

% The list of subjects to be included in the study.
nSub=20;
userOptions.subjectNames = cellfun(@(x) sprintf('subject%d',x),num2cell(1:nSub),'uniformoutput',0);

% Text labels which may be attached to the conditions for MDS plots.
nCond=64;
userOptions.conditionLabels = cellfun(@(x) sprintf('condition%d',x),num2cell(1:nCond),'uniformoutput',0);
[userOptions.alternativeConditionLabels{1:nCond}] = deal(' ');
userOptions.useAlternativeConditionLabels = true;

% Which distance measure to use when calculating first-order RDMs.
userOptions.distance = 'Correlation';

% Which similarity-measure is used for the second-order comparison.
userOptions.distanceMeasure = 'Spearman';
userOptions.RDMcorrelationType=userOptions.distanceMeasure; % different names for same thing, depending on function??

% What criterion shoud be minimised in MDS display?
userOptions.criterion = 'metricstress';

% How many permutations should be used to test the significance of the fits?
userOptions.significanceTestPermutations = 1000; % (10,000 highly recommended); this is overridden by .nRandomisations??

% Bootstrap options
userOptions.nResamplings = 1000;
userOptions.resampleSubjects = true;
userOptions.resampleConditions = true;

% plotting options
userOptions.rankTransform = false;
userOptions.colourScheme = jet(64);
userOptions.rubberbands = true;
userOptions.plottingStyle = 2; % ?
userOptions.conditionColours = kron([1 0 0; 0 1 0; 0 0 1; 1 .7 0], ones(16,1)); % here the 64 conditions are allocated to 4 groups
userOptions.convexHulls = [ones(1,16) 2*ones(1,16) 3*ones(1,16) 4*ones(1,16)]; % here the 64 conditions are allocated to 4 groups
userOptions.RoIColor = [0 0 1];
userOptions.ModelColor = [0 1 0];

% How should figures be outputted?
userOptions.displayFigures = true;
userOptions.saveFiguresPDF = false;
userOptions.saveFiguresFig = false;
userOptions.saveFiguresPS = false;
userOptions.saveFiguresEps = false;

%% Generate a simulationOptions structure.
simulationOptions = simulationOptions_demo_SL();

%% set some searchlight options
searchlightOptions.monitor = false; % must be false, because a function is missing
searchlightOptions.fisher = true;
searchlightOptions.nSessions = 1;
searchlightOptions.nConditions = simulationOptions.nConditions;

userOptions.searchlightRadius = 9; % mm?
userOptions.voxelSize = [3 3 3];

load(fullfile(here,'sampleMask_org.mat')); % load brain mask 'm'
brainmask=m;
brainmask(:,:,[1:15 23:end])=0; % djm: reduce analysis volume to speed things up
load(fullfile(here,'anatomy.mat'));% load resliced structural image 'anatVol'

%% a simple 2-category model RDM
models = constructModelRDMs(modelRDMs_SL_sim, userOptions);

%% simulate the data per subject
fprintf('simulating fullBrain volumes for %d subjects (expect ~3 s per subject)\n',nSub); t0=tic;
fMRI_sub=cell(1,nSub);
[B_true,effectmask,Y_true, fMRI_sub{1:nSub}] = simulateClusteredfMRIData_fullBrain(simulationOptions);
fprintf('(Took %.0f s for %d subjects)',toc(t0),nSub)

%% display the design matrix
selectPlot(1);
subplot(321);imagesc(fMRI_sub{1}.X);axis square
ylabel('\bfscans');xlabel('\bfregressors')
title('\bfdesign matrix')

subplot(322);plot(fMRI_sub{1}.X(:,12:14))
xlabel('scans');title('\bfregressors for 3 example conditions')

%% display simulated RDM
subplot(323);
image(scale01(rankTransform_equalsStayEqual(squareform(pdist(fMRI_sub{1}.groundTruth)),1)),'CDataMapping','scaled','AlphaData',~isnan(squareform(pdist(fMRI_sub{1}.groundTruth))));
axis square off
title('\bfsimulated ground truth RDM')
showRDMs(pdist(fMRI_sub{1}.groundTruth),[1 3 2 3])

%% display model RDM
subplot(325);
image(scale01(rankTransform_equalsStayEqual(models(1).RDM,1)),'CDataMapping','scaled','AlphaData',~isnan(models(1).RDM));
axis square off
colormap(RDMcolormap)
title('\bftested model RDM')

%% display searchlight
relRoi = sphericalRelativeRoi(userOptions.searchlightRadius,userOptions.voxelSize);
nVox_searchlight = size(relRoi,1);
showVoxObj(relRoi+repmat(simulationOptions.effectCen,[nVox_searchlight,1]), 1, [3 2 6], [], [], [0   0.5   0.3  1 0.4]);
title(['\bf searchlight with ',num2str(nVox_searchlight),' voxels']); drawnow
handleCurrentFigure(fullfile(here,'DEMO4','SLsimulationSettings'),userOptions);

%% compute the correlation maps per subject (expect ~10s per subject)
rMaps=cell(1,nSub);
for subI = 1:nSub
        fprintf('\n computing correlation maps for subject %d',subI)
        [rs, ps, ns] = searchlightMapping_fMRI(fMRI_sub{subI}.B', models, brainmask, userOptions, searchlightOptions);
        % This is the important function; it wants the beta maps B [voxels x conditions]
        % along with the model RDM, mask, and options
        gotoDir(userOptions.rootPath, 'Maps');
        save(['rs_subject',num2str(subI),'.mat'],'rs');
        rMaps{subI}=rs;
        clear rs
        fprintf('.')
end


%% test correlation maps across subjects
fprintf('\n Running group FRX stats')
for modelI = 1:numel(models)
        thisModelSims=nan([size(brainmask) nSub]);
        for subI = 1:nSub
                thisRs = rMaps{subI};
                thisModelSims(:,:,:,subI) = thisRs(:,:,:,modelI);
        end
        % obtain a pMaps from applying a 1-sided signrank test and also t-test to
        % the model similarities:
        p1=nan(size(brainmask));
        p2=nan(size(brainmask));
        for x=1:size(thisModelSims,1)
                for y=1:size(thisModelSims,2)
                        for z=1:size(thisModelSims,3)
                                if brainmask(x,y,z) == 1
                                        [h, p1(x,y,z)] = ttest(squeeze(thisModelSims(x,y,z,:)),0,0.05,'right');
                                        [p2(x,y,z)] = signrank_onesided(squeeze(thisModelSims(x,y,z,:)));
                                end
                        end
                end
                fprintf('.')
        end
        % apply FDR correction
        pThrsh_t = FDRthreshold(p1,0.05,brainmask);
        pThrsh_sr = FDRthreshold(p2,0.05,brainmask);
        %p_bnf = 0.05/sum(mask(:));
        
        % mark the suprathreshold voxels in yellow
        supraThreshMarked_t = (p1 <= pThrsh_t);
        supraThreshMarked_sr = (p2 <= pThrsh_sr);
        
        % make brain volume to view as slices
        brainVol = addRoiToVol(map2vol(anatVol),mask2roi(brainmask),[1 0 0],2);
        
        % display the location where the effect was inserted (in green):
        brainVol_effectLoc = addBinaryMapToVol(brainVol,effectmask.*brainmask,[0 1 0]);
        showVol(brainVol_effectLoc,'simulated effect [green]',2);
        handleCurrentFigure(fullfile(here,'DEMO4','results_DEMO4_simulatedEffectRegion'),userOptions);
        
        % display the FDR-thresholded maps on a sample anatomy (signed rank test) :
        brainVol_sr = addBinaryMapToVol(brainVol,supraThreshMarked_sr.*brainmask,[1 1 0]);
        showVol(brainVol_sr,'signrank, E(FDR) < .05',3)
        handleCurrentFigure(fullfile(here,'DEMO4','results_DEMO4_signRank'),userOptions);
        
        % display the FDR-thresholded maps on a sample anatomy (t-test) :
        brainVol_t = addBinaryMapToVol(brainVol,supraThreshMarked_t.*brainmask,[1 1 0]);
        showVol(brainVol_t,'t-test, E(FDR) < .05',4)
        handleCurrentFigure(fullfile(here,'DEMO4','results_DEMO2_tTest'),userOptions);
end

keyboard

return

