function DEMO3_LDt_sim_djm()
%__________________________________________________________________________
% Copyright (C) 2010 Medical Research Council
%
% modified by djm, 4/2017, 5/9/22
% Similar to Figure S5 from paper: using LD-t as a dissimilarity measure

%%%%%%%%%%%%%%%%%%%%
%% Initialisation %%
%%%%%%%%%%%%%%%%%%%%

close all
dbstop if error

% add paths to toolbox:
here=fileparts(which(mfilename));
addpath(genpath(fileparts(here)));
import rsa.*
import rsa.rdm.*
import rsa.fig.*
import rsa.stat.*
import rsa.sim.*
import rsa.fmri.*

% set the output directory:
userOptions.rootPath = fullfile(here,'DEMO3');
userOptions.analysisName = 'DEMO3';
if ~exist(userOptions.rootPath,'dir'), mkdir(userOptions.rootPath); end

% Generate/populate the userOptions structure
% djm: I'll do it explicitly here, rather than calling a separate function.
% userOptions = projectOptions_demo();

% The path leading to where the scans are stored (not including subject-specific identifiers).
% "[[subjectName]]" should be used as a placeholder to denote an entry in userOptions.subjectNames
% "[[betaIdentifier]]" should be used as a placeholder to denote an output of betaCorrespondence.m if SPM is not being used; or an arbitrary filename if SPM is being used.
userOptions.betaPath = fullfile(here,'demoTest','[[subjectName]]','[[betaIdentifier]]');

% The list of subjects to be included in the study.
nSub=20;
userOptions.subjectNames = cellfun(@(x) sprintf('subject%d',x),num2cell(1:nSub),'uniformoutput',0);

% Text labels which may be attached to the conditions for MDS plots.
nCond=64;
userOptions.conditionLabels = cellfun(@(x) sprintf('condition%d',x),num2cell(1:nCond),'uniformoutput',0);
[userOptions.alternativeConditionLabels{1:nCond}] = deal(' ');
userOptions.useAlternativeConditionLabels = true;

% Which distance measure to use when calculating first-order RDMs.
userOptions.distance = 'Correlation'; % this won't be used in this demo!

% Which similarity-measure is used for the second-order comparison.
userOptions.distanceMeasure = 'Spearman';
userOptions.RDMcorrelationType=userOptions.distanceMeasure; % different names for same thing, depending on function??

% What criterion shoud be minimised in MDS display?
userOptions.criterion = 'metricstress';

% How many permutations should be used to test the significance of the fits?
userOptions.significanceTestPermutations = 1000; % (10,000 highly recommended); this is overridden by .nRandomisations??

% Bootstrap options
userOptions.nResamplings = 1000;
userOptions.resampleSubjects = true;
userOptions.resampleConditions = true;

% plotting options
userOptions.rankTransform = false;
userOptions.colourScheme = jet(64);
userOptions.rubberbands = true;
userOptions.plottingStyle = 2; % ?
userOptions.conditionColours = kron([1 0 0; 0 1 0; 0 0 1; 1 .7 0], ones(16,1)); % here the 64 conditions are allocated to 4 groups
userOptions.convexHulls = [ones(1,16) 2*ones(1,16) 3*ones(1,16) 4*ones(1,16)]; % here the 64 conditions are allocated to 4 groups
userOptions.RoIColor = [0 0 1];
userOptions.ModelColor = [0 1 0];

% How should figures be outputted?
userOptions.displayFigures = true;
userOptions.saveFiguresPDF = false;
userOptions.saveFiguresFig = false;
userOptions.saveFiguresPS = false;
userOptions.saveFiguresEps = false;

%% Generate a simulationOptions structure.
simulationOptions = simulationOptions_demo_LDt();
% again, a 7x7x7 'brain', with 64 conditions in 4 nested categories

%% simulate data for two sessions per subject and compute subject-specific LDt RDMs:
for subI = 1:nSub
        fprintf('simulating data and computing LD-t values for subject %d \n',subI)
        [B_true,fMRI_a,fMRI_b] = simulateClusteredfMRIData(simulationOptions);
        RDM_fdtFolded_ltv = fisherDiscrTRDM(fMRI_a.X,fMRI_a.Y,fMRI_b.X,fMRI_b.Y); 
        % this is the important function for calculating the LDt; 
        % it needs the data Y [time x voxel] and design matrix X [time x condition] for each of two runs
        
        RDMs(subI).RDM = squareform(RDM_fdtFolded_ltv);% diagonals will contain zeros
        RDMs(subI).name = ['LDAtRDM | subject',num2str(subI)];
        RDMs(subI).color = [1 0 0];
end

%% compute the subject-averaged LDAtRDM
averageRDMs_LDt = averageRDMs_subjectSession(RDMs, 'subject');

%% display the average LDAtRDM
averageRDMs_LDt.name = 'subject-average LD-tRDM';
showRDMs(averageRDMs_LDt,2,0)
handleCurrentFigure([userOptions.rootPath,filesep,'demo_LDAtRDM_simulation_groupAverageRDM'],userOptions);
subplot(1,2,1); caxis(max(abs(caxis))*[-1 1]); % djm: make colour scale symmetrical about zero
subplot(1,2,2); caxis(max(abs(caxis))*[-1 1]);

%% random effects analysis
% can be conducted per cell of the RDM, comparing the LDt to zero
rdms = unwrapRDMs(RDMs);% nCond x nCond x nSubj
p_t = ones(nCond,nCond);
p_sr = ones(nCond,nCond);
for condI = 1:nCond
        for condJ = condI+1:nCond
                [~, p_t(condI,condJ)] = ttest(squeeze(rdms(condI,condJ,:)),0,0.05,'right');
                [p_sr(condI,condJ)] = signrank_onesided(squeeze(rdms(condI,condJ,:)));
        end
end

%% compute the thresholds and display
thresh_uncorr = 0.05;
nTests = nCond*(nCond-1)/2;
thresh_fdr_t = FDRthreshold(p_t,thresh_uncorr);
thresh_fdr_sr = FDRthreshold(p_sr,thresh_uncorr);
thresh_bnf = thresh_uncorr/nTests;
selectPlot(5);
subplot(231);image_thr(p_sr,thresh_uncorr)
axis square off;
title('\bf SignRank p < 0.05 (uncorr.)')

subplot(232);image_thr(p_sr,thresh_fdr_sr)
axis square off;
title('\bf SignRank p < 0.05 (FDR)')

subplot(233);image_thr(p_sr,thresh_bnf)
axis square off;
title('\bf SignRank p < 0.05 (Bonferroni)')

subplot(234);image_thr(p_t,thresh_uncorr)
axis square off;
title('\bf Ttest p < 0.05 (uncorr.)')

subplot(235);image_thr(p_t,thresh_fdr_t)
axis square off;
title('\bf Ttest p < 0.05 (FDR)')

subplot(236);image_thr(p_t,thresh_bnf)
axis square off;
title('\bf Ttest p < 0.05 (Bonferroni)')

addHeading('random effect analysis, subjects as random effects')
handleCurrentFigure([userOptions.rootPath,filesep,'demo_LDAtRDM_simulation_subjectRFX'],userOptions);

return

