function DEMO1_RSA_ROI_simulatedAndRealData_djm()

% DEMO1_RSA_ROI_simulatedAndRealData
%
% This function computes the results in the main Figures (2,3,4,5) of
% Nili et al. (PLoS Comp Biol 2013)
%
% modified by djm, 4/2017, 5/9/22

close all
dbstop if error

%% add paths to toolbox:
here=fileparts(which(mfilename));
addpath(genpath(fileparts(here)));
import rsa.*
import rsa.rdm.*
import rsa.fig.*
import rsa.stat.*

%% fill in default options:
userOptions = defineUserOptions;

%% set random number generator for reproducability
rng(1);

%% set the output directory:
userOptions.rootPath = fullfile(here,'DEMO1');
userOptions.analysisName = 'DEMO1';
if ~exist(userOptions.rootPath,'dir'); mkdir(userOptions.rootPath); end
userOptions.resultsPath = userOptions.rootPath;

%% over-ride any defaults (djm)
% userOptions.tightInset=true;
% userOptions.saveFiguresPDF=0; % if crashing for any reason

%% load monkey IT RDM and category definitions from Kriegeskorte et al. (Neuron 2008)
% (can think of monkey RDM as model for human RDMs)
data92=load(fullfile(here,'92imageData','faceAnimateInaniClustersRDM.mat'));
% data92.RDMs_mIT_hIT_fig1 is a structure containing two wrapped RDMs:
myModelRDMs.monkeyIT=squareRDMs(data92.RDMs_mIT_hIT_fig1(1).RDM); % monkey IT (the 2nd entry is from human fMRI data)
% this is equivalent to: stripNsquareRDMs(data92.RDMs_mIT_hIT_fig1(1)); 
% or:                          squareform(data92.RDMs_mIT_hIT_fig1(1).RDM);

% data92 also contains category codes, stimulus images, and various bare
% model RDMs, e.g.:
myModelRDMs.Face_Ani_Inani=data92.faceAnimateInaniClustersRDM; % this is a simple nested categorical model RDM, already "stripped and squared"

%% load and view some real human subject fMRI RDMs (top panel of figure 5 from paper)
temp=load(fullfile(here,'92imageData','92_brainRDMs.mat'));
data92.RDMs_hIT_bySubjectSession=temp.RDMs;
% data92.RDMs_hIT_bySubjectSession contains wrapped RDMs from 4 subjects with 2 sessions each
% I.e. the array is organised as [roi x subject x session].

% average across sessions:
data92.RDMs_hIT_bySubject = averageRDMs_subjectSession(data92.RDMs_hIT_bySubjectSession, 'session');

% display RDMs per subject:
figure(1); clf(1);
showRDMs(data92.RDMs_hIT_bySubject,1); % djm: added colormap(colourScheme); on line 73 of showRDMs
set(1,'name','Real subject RDMs')

% can use this to print the figure; see above userOptions for changing default figure formats to print
handleCurrentFigure(fullfile(userOptions.rootPath,'subjectRDMs_hIT_fMRI'),userOptions);


%% simulate multiple subjects' noisy RDMs (top of figure 4 from paper)

nSubjects=12;
subjectPatternNoiseStd=1;

% load reconstructed patterns for simulating models
data92.patterns=load(fullfile(here,'92imageData','simTruePatterns.mat'));
[nCond, nDim]=size(data92.patterns.simTruePatterns); % [92 images x 100 voxels]

subjectRDMs=nan(nCond,nCond,nSubjects);

for subjectI=1:nSubjects
        % add noise to underlying pattern:
        patterns_cSubject=data92.patterns.simTruePatterns2+subjectPatternNoiseStd*randn(nCond,nDim);
        % calculate RDM:
        subjectRDMs(:,:,subjectI)=squareRDMs(pdist(patterns_cSubject,userOptions.distance));
end

% calculate the average across subjects
avgRDM=wrapAndNameRDMs(mean(subjectRDMs,3),{'subject-averaged RDM'});

% display simulated RDMs per subject, and average:
showRDMs(concatRDMs_unwrapped(subjectRDMs,avgRDM),2); % djm: note bug in concatRDMs_unwrapped.m: change unwrap to unwrapRDMs
set(2,'name','Simulated subject RDMs, and average')
handleCurrentFigure(fullfile(userOptions.rootPath,'simulatedSubjAndAverage'),userOptions);

keyboard % 

% to include image icons:
% showRDMs(avgRDM,2,0,[],0, [], struct('images',data92.stimuli_92objs))

%% define various model RDMs (bottom of figure 4 from paper)

% categorical split by animacy:
myModelRDMs.Ani_Inani=categoricalRDM(data92.categoryVectors(:,1),[],false);

% split by multiple categories:
Face_Body_Nat_Artif=5:8;
myModelRDMs.Face_Body_Nat_Artif=categoricalRDM(data92.categoryVectors(:,Face_Body_Nat_Artif),[],false);

% load behavioural RDM from Mur et al. (Frontiers Perc Sci 2013)
temp=load(fullfile(here,'92imageData','92_behavRDMs.mat'));
data92.behav=temp.rdms_behav_92;
myModelRDMs.SimilarityJudgements=mean(stripNsquareRDMs(data92.behav),3);

% % create modelRDMs with different levels of noise? Not used in this demo?
% patternDevStds=linspace(bestModelPatternDeviationStd,worstModelPatternDeviationStd,nModelGrades);
%
% for gradedModelI=1:nModelGrades
%     patterns_cGradedModel=data92.patterns.simTruePatterns2+patternDevStds(gradedModelI)*randn(nCond,nDim);
%     myModelRDMs.(sprintf('TrueModelWithNoise%.0f',patternDevStds(gradedModelI)))=squareRDMs(pdist(patterns_cGradedModel,userOptions.distance));
% end
myModelRDMs.TrueModel=squareform(pdist(data92.patterns.simTruePatterns2,userOptions.distance));

% load V1 model and HMAX model with natural image patches from Serre et al. (Computer Vision and Pattern Recognition 2005)
data92.V1Models=load(fullfile(here,'92imageData','rdm92_V1model.mat'));
myModelRDMs.V1model=data92.V1Models.rdm92_V1model;
myModelRDMs.HMAX_2005=data92.V1Models.rdm92_HMAXnatImPatch;

% load RADON and silhouette models and human early visual RDM
data92.OtherModels=load(fullfile(here,'92imageData','92_modelRDMs.mat'));
myModelRDMs.humanEarlyVisual=data92.OtherModels.Models(4).RDM;
myModelRDMs.silhouette=data92.OtherModels.Models(7).RDM;
myModelRDMs.RADON=data92.OtherModels.Models(8).RDM;


%% wrap, name, and display the model RDMs

modelnames=fieldnames(myModelRDMs);
for m=1:length(modelnames)
        modelRDMs(m)=wrapAndNameRDMs(myModelRDMs.(modelnames{m}),modelnames(m));
end

showRDMs(modelRDMs,3);
set(3,'name','Model RDMs')
handleCurrentFigure(fullfile(userOptions.rootPath,'allModels'),userOptions);

keyboard % pause to view RDMs and return to slides to consider visualisations


%% recreate figure 2 of paper (with Shepard plots instead of RDMs)...
% Notes:
% With metric stress the disparities in the Shepard plot are a straight line, i.e. there's no
% monotonic transform of the dissimilarities as with nonmetric stress.
% Change this with userOptions.criterion. Nonmetric stress may not make
% sense with ties and very 'categorical' data. 'Strain' would assume
% dissimilarities are Euclidean.
% In these data, the relation of the distances to the disparities suggest that
% small dissimilarities are being underestimated. The points are also
% pushed towards a circle/hypersphere, this can happen when points are
% random or equally far from each other in high-d space, which can't be well
% represented in 2d.
%
% The orientation of the MDS solutions is arbitrary, but could be
% aligned with Procrustes rotation.
%

%% assign 'condition colours' to the categories of interest
categoryCols=[0 0 0
        0 0 0
        0 0 0
        0 0 0
        1 0.5 0
        1 0 0
        0 1 0
        0 0.5 1];

for condI = 1:92
        for catI = 1:numel(Face_Body_Nat_Artif)
                if data92.categoryVectors(condI,Face_Body_Nat_Artif(catI))
                        userOptions.conditionColours(condI,:) = categoryCols(Face_Body_Nat_Artif(catI),:);
                end
        end
end

fig=4;
figure(fig); set(fig,'name','Respresentational space of data'); clf;

%%%%%%% activity pattern MDS of true model

%  MDSConditions(modelRDMs(ismember(modelnames,{'TrueModel'})), userOptions,...
%      struct('titleString','ground-truth MDS','fileName','trueRDM_MDS','figureNumber',6));

figureMDSArrangement(modelRDMs(ismember(modelnames,{'TrueModel'})), userOptions,...
        struct('titleString','ground-truth MDS','fileName','trueRDM_MDS',...
        'textLabels',{userOptions.conditionLabels},'dotColours',userOptions.conditionColours,...
        'figI_textLabels',[fig 3 3 3+1],'figI_shepardPlots',[fig 3 3 0+1])); % djm: reduced title fontsize to 6
% note alternative functions with overlapping functionality

% Dendrogram of true model
dendrogramConditions(modelRDMs(ismember(modelnames,'TrueModel')), userOptions,...
        struct('titleString', 'Dendrogram of the ground truth RDM', 'figureNumber', [fig 3 3 6+1]));

%%%%%%% MDS of single subject RDM (e.g. simulated subject1), noisier
figureMDSArrangement(wrapAndNameRDMs(subjectRDMs(:,:,1),{'single-subject RDM'}), userOptions,...
        struct('titleString','simulated data: one subject','fileName','sample subject MDS',...
        'textLabels',{userOptions.conditionLabels},'dotColours',userOptions.conditionColours,...
        'figI_textLabels',[fig 3 3 3+2],'figI_shepardPlots',[fig 3 3 0+2]));

% Dendrogram of  single subject RDM
dendrogramConditions(wrapAndNameRDMs(subjectRDMs(:,:,1),{'single-subject RDM'}), userOptions,...
        struct('titleString', 'Dendrogram of a single-subject RDM', 'figureNumber', [fig 3 3 6+2]));

%%%%%%% MDS of subject-averaged RDM
figureMDSArrangement(avgRDM, userOptions,...
        struct('titleString','simulated data: group average','fileName','trueRDM_MDS',...
        'textLabels',{userOptions.conditionLabels},'dotColours',userOptions.conditionColours,...
        'figI_textLabels',[fig 3 3 3+3],'figI_shepardPlots',[fig 3 3 0+3]));

% Dendrogram of subject-averaged RDM
dendrogramConditions(avgRDM, userOptions,...
        struct('titleString', 'Dendrogram of the subject-averaged RDM', 'figureNumber', [fig 3 3 6+3]));


%% recreate figure 3 of paper (+ Shepard plots): 2nd order RDM correlation matrix and MDS
fig=5; figure(fig); clf(fig);

%userOptions.distanceMeasure='Kendall_taua'; % slow!
userOptions.distanceMeasure='Spearman';

userOptions.RDMcorrelationType=userOptions.distanceMeasure; % different names for same thing, depending on function??
% Confusingly, both of these seem to refer to distances between RDMs, while
% userOptions.distance refers to the distance between patterns within an
% RDM

%%%%% using the high-level functions

% pairwiseCorrelateRDMs({avgRDM, modelRDMs}, userOptions, struct('figureNumber', [5 1 2 1],'fileName','RDMcorrelationMatrix'));
%
% MDSRDMs({avgRDM, modelRDMs}, userOptions, struct('titleString', 'MDS of different RDMs', 'figureNumber', 5,'fileName','2ndOrderMDSplot'));

%%%%%% using the lower-level functions
allRDMs=[avgRDM, modelRDMs];
RDMRDM=1-RDMCorrMat(allRDMs,fig,userOptions.RDMcorrelationType);
showRDMs(1-RDMRDM,[5 1 3 1],false,[-1 1],false); % no rank transform; no separate axis colourbar
colormap(colorScale([0 0 1; .5 .5 .5; 1 0 0],256));
colorbar('southoutside')
for RDMI=1:length(allRDMs)
        text(RDMI,RDMI,allRDMs(RDMI).name,'HorizontalAlignment','center','FontWeight','bold','interpreter','none','Color',allRDMs(RDMI).color,'FontUnits','normalized','FontSize',1/(length(allRDMs)*2));
end
title(['RDM correlation matrix (',userOptions.RDMcorrelationType,')'],'FontWeight','bold','interpreter','none');

figureMDSArrangement(RDMRDM, userOptions,...
        struct('titleString','MDS of different RDMs','fileName','2ndOrderMDSplot',...
        'textLabels',{{allRDMs.name}},'dotColours',vertcat(allRDMs.color),...
        'figI_textLabels',[fig 1 3 2],'figI_shepardPlots',[fig 1 3 3],...
        'rubberbandGraphPlot',true));
%%%%%%%

keyboard % pause to consider statistical inference

%% statistical inference (the first figure recreates the main panel of figure 4 from paper)
% (The 2nd figure shows pairwise model comparisons as matrices with various significance thresholds)
userOptions.RDMrelatednessTest = 'subjectRFXsignedRank'; % for how each model RDM relates to data RDMs
userOptions.RDMrelatednessThreshold = 0.05;
userOptions.RDMrelatednessMultipleTesting = 'FDR';
userOptions.candRDMdifferencesTest = 'subjectRFXsignedRank'; % for how model pairs compare in explaining data RDMs
userOptions.candRDMdifferencesThreshold = 0.05;
userOptions.candRDMdifferencesMultipleTesting = 'FDR';
userOptions.plotpValues = '=';
userOptions.barsOrderedByRDMCorr=true;
userOptions.figureIndex = [6 7];
userOptions.figure1filename = 'compareRefRDM2candRDMs_barGraph_simulatedITasRef';
userOptions.figure2filename = 'compareRefRDM2candRDMs_pValues_simulatedITasRef';
stats_p_r=compareRefRDM2candRDMs(subjectRDMs, num2cell(modelRDMs), userOptions);
% This function is important, and has lots of help at the top.
% The first input is a single reference RDM (any format), or a stack of samples of the reference RDM.
% Second input is cell array, because each cell could contain stacked RDMs.

keyboard % pause; note next section will be slow due to bootstrap...

%% Finally: real fMRI data (human IT RDM from Kriegeskorte et al. (Neuron 2008) as the reference RDM
% (recreate main panel of figure 5 from paper, but using Spearman)
% The bootstrap test for RDM differences is slow...

%userOptions.distanceMeasure='Kendall_taua'; % slow!
userOptions.distanceMeasure='Spearman';
userOptions.RDMcorrelationType=userOptions.distanceMeasure; % different names for same thing, depending on function??

userOptions.RDMrelatednessTest = 'randomisation'; % changed to fixed effects as there are only 4 subjects
userOptions.RDMrelatednessThreshold = 0.05;
userOptions.RDMrelatednessMultipleTesting = 'FDR'; % 'none'; 'FWE'
userOptions.candRDMdifferencesTest = 'conditionRFXbootstrap'; % changed to condition bootstrap
userOptions.candRDMdifferencesThreshold = 0.05;
userOptions.candRDMdifferencesMultipleTesting = 'FDR';
userOptions.plotpValues = '*';
userOptions.barsOrderedByRDMCorr=true;
userOptions.nRandomisations = 100; % now needed
userOptions.nBootstrap = 100; % default 10000, not 1000 as stated in help? % 100 is too few, but still
userOptions.figureIndex = [8 9];
userOptions.figure1filename = 'compareRefRDM2candRDMs_barGraph_hITasRef';
userOptions.figure2filename = 'compareRefRDM2candRDMs_pValues_hITasRef';
stats_p_r=compareRefRDM2candRDMs(data92.RDMs_hIT_bySubject, num2cell(modelRDMs(~ismember(modelnames,'TrueModel'))), userOptions);

% confusingly, the following settings in the userOptions seem to be ignored??
%    userOptions.significanceTestPermutations;
%    userOptions.nResamplings;
%    userOptions.resampleSubjects;
%    userOptions.resampleConditions;

keyboard

return
