% 2D transcranial ultrasound example prepared for the FUN21 conference.
%
% This script provides a simple example of using k-Wave to perform a
% transcranial ultrasound simulation in 2D. It requires a few things to
% run. Most of these are included in the zip file, but for reference:
%
%   1. MATLAB. Any recent version should be fine, although I haven't
%   checked rigorously.
%
%   2. MATLAB Image Processing Toolbox. Needed if you want to plot the
%   edges of the skull while running the simulation. If you don't have this
%   toolbox, no problems, just set 'DisplayMask' to skull_mask, or remove
%   this line of code.
%
%   3. k-Wave V1.3. This can be downloaded from
%   http://www.k-wave.org/download.php.
% 
%   4. Alpha version of the kWaveArray class. This can be downloaded from
%   http://www.k-wave.org/forum/topic/alpha-version-of-kwavearray-off-grid-sources
%
%   5. MNI template CT. This can be downloaded from
%   https://github.com/neurolabusc/Clinical/tree/master/Tutorial/high_res
%   (the file is called scct_unsmooth.nii). The reference for the data is
%   https://doi.org/10.1016/j.neuroimage.2012.03.020.
%
%   6. Nifti toolbox from the file exchange
%   https://uk.mathworks.com/matlabcentral/fileexchange/8797-tools-for-nifti-and-analyze-image.
%   There are some built in nifti tools in recent versions of MATLAB, but
%   they don't seem to apply the scl_slope and scl_inter fields correctly,
%   so the Hounsfield units come out wrong. 
%
% I've assumed some familiarity with both MATLAB and k-Wave. If you're new
% to MATLAB / programming, I'd suggest going through some of the excellent
% tutorials on the web and included in MATLAB. If you're new to k-Wave, I'd
% recommend spending a few afternoons going through the examples and
% reading the documentation.
%
% The example loads the template CT image from the nifti file (see download
% link above), and selects a 2D slice. I use 2D to make the simulations
% fast and easy to visualise - the same steps can be used in 3D. Note, I
% strongly recommend you don't run any real simulations in 2D. Because of
% the differences in symmetry and geometric spreading, amoung other things,
% the focal gain will be incorrect, and standing wave effects /
% interference patterns will be much more pronounced.
%
% The skull properties are mapped from CT Hounsfield units using the
% porosity approach described in
% (https://doi.org/10.1088/0031-9155/54/9/001). There are lots of flavours
% and variations to this, I've just picked one. The "best" one to use is
% still an active research topic.
%
% PLEASE NOTE: The script resets and updates the MATLAB path, clears the
% workspace, and closes all open figures. If you don't want this, then
% modify accordingly.
%
% author: Bradley Treeby
% date: 31 August 2021
% last update: 1 September 2021

% clear paths to avoid conflicts and add relevant ones
restoredefaultpath;
addpath('Nifti-Toolbox');
addpath('k-Wave');
addpath('kWaveArray');

% close 
close all hidden;
clearvars;

% =========================================================================
% LOAD AND PROCESS CT IMAGE
% =========================================================================

% load template CT image (1 mm isotropic)
ct_3D = load_nii('scct_unsmooth.nii');
dx = 1e-3;

% select slice
ct_2D = ct_3D.img(2:end, 2:end, 70);

% resample image here if desired - I'd recommend at least 6 grid points per
% wavelength (e.g., see https://doi.org/10.1121/1.4976339)

% pad the posterior direction to allow room for the transducer to fit on
% the grid
ct_2D = expandMatrix(ct_2D, [0, 0, 10, 0]);

% get grid size from the padded image
[Nx, Ny] = size(ct_2D);

% segment the skull based on the Hounsfield units
skull_mask = (ct_2D > 300);

% calculate porosity within the skull mask (well, actually 1 - porosity,
% but it saves some operations this way)
skull_porosity = zeros(Nx, Ny);
skull_porosity(skull_mask) = ct_2D(skull_mask);
skull_porosity = skull_porosity ./ max(skull_porosity(:));

% use porosity to map medium properties based on a linear relationship
% between the porosity and the medium properties
medium.sound_speed_ref = 1500;
medium.sound_speed = 1500 + skull_porosity * 1600;
medium.density = 1000 + skull_porosity * 1200;

% assume a homogeneous attenuation across the skull with 0.6 dB/MHz/cm in
% the brain / background and 8 dB/MHz/cm in the skull
alpha_coeff_true = 0.6 + skull_mask * 7.4;
alpha_power_true = ones(Nx, Ny);

% account for actual absorption behaviour in k-Wave, which varies when high
% absorption is used (see https://doi.org/10.1121/1.4894790)
medium.alpha_coeff = fitPowerLawParamsMulti(alpha_coeff_true, alpha_power_true, medium.sound_speed, 250e3, 2, true);
medium.alpha_power = 2;

% create grid
cfl = 0.3;
t_end = 300e-6;
kgrid = kWaveGrid(Nx, dx, Ny, dx);
kgrid.makeTime(medium.sound_speed, cfl, t_end);

% =========================================================================
% SETUP TRANSDUCER
% =========================================================================

% create empty array
karray = kWaveArray;

% add single-element focused bowl
position = [5e-3, -0.11];
radius = 64e-3;
diameter = 64e-3;
focus_pos = [-5e-3, 0];
karray.addArcElement(position, radius, diameter, focus_pos);

% create driving signal
freq = 250e3;
amp = 100e3;
input_signal = createCWSignals(kgrid.t_array, freq, amp, 0);

% =========================================================================
% RUN SIMULATION
% =========================================================================

% create distributed source signal (in k-Wave V1.3, the kWaveArray object
% can't be passed directly to the simulation functions, so we use these
% helper methods to generate the standard source mask and pressure input)
source.p_mask = karray.getArrayBinaryMask(kgrid);
source.p = karray.getDistributedSourceSignal(kgrid, input_signal);

% record just the maximum amplitudes (for continuous wave simulations, a
% more accurate way is to set the time step to an integer number of points
% per period, record for an integer number of periods in steady state, and
% then extract the amplitudes spectrally, but taking the maximum is an ok
% approximation)
sensor.record = {'p_max_all'};

% run simulation, placing the perfectly matched layer outside the grid, and
% choosing the pml size automatically to give small prime factors
sensor_data = kspaceFirstOrder2D(kgrid, medium, source, sensor, ...
    'PMLInside', false, ...
    'PMLSize', 'auto', ...
    'PlotPML', false, ...
    'PlotScale', amp * [-1, 1], ...
    'DisplayMask', edge(skull_mask, 'Canny'), ...
    'DataCast', 'single');

% =========================================================================
% PLOT MEDIUM PROPERTIES
% =========================================================================

% plot slice
figure;
subplot(2, 3, 1);
imagesc(kgrid.y_vec, kgrid.x_vec, ct_2D);
axis image;
colormap(gray);
colorbar;
karray.plotArray(false);
title('CT Image');

% plot mask
subplot(2, 3, 2);
imagesc(kgrid.y_vec, kgrid.x_vec, skull_mask);
axis image;
colormap(gray);
colorbar;
karray.plotArray(false);
title('Skull Mask');

% plot porosity
subplot(2, 3, 3);
imagesc(kgrid.y_vec, kgrid.x_vec, skull_porosity);
axis image;
colormap(gray);
colorbar;
karray.plotArray(false);
title('Skull Porosity');

% plot sound speed
subplot(2, 3, 4);
imagesc(kgrid.y_vec, kgrid.x_vec, medium.sound_speed);
axis image;
colormap(gray);
colorbar;
karray.plotArray(false);
title('Sound Speed');

% plot density
subplot(2, 3, 5);
imagesc(kgrid.y_vec, kgrid.x_vec, medium.density);
axis image;
colormap(gray);
colorbar;
karray.plotArray(false);
title('Density');

% plot absorption
subplot(2, 3, 6);
imagesc(kgrid.y_vec, kgrid.x_vec, medium.alpha_coeff);
axis image;
colormap(gray);
colorbar;
karray.plotArray(false);
title('Absorption');

scaleFig(2, 1);

%sgtitle('Material Properties Mapped From CT');
drawnow;

% =========================================================================
% PLOT SIMULATION RESULTS
% =========================================================================

figure;
overlayPlot(kgrid.y_vec, kgrid.x_vec, ct_2D, 1e-3 * sensor_data.p_max_all, ...
    'ColorBar', true, ...
    'LogComp', false, ...
    'ColorBarTitle', '[kPa]', ...
    'PlotScale', [0, 300]);
karray.plotArray(false);
title('Simulated Pressure Field');